const React = require('react');
const { Helmet } = require('react-helmet');
const { StaticRouter, Route } = require('react-router-dom');
const { renderToStaticMarkup, renderToString } = require('react-dom/server');
const { QueryParamProvider } = require('use-query-params');

const htmlescape = require('htmlescape');
const cfg = require('yandex-cfg');

const { getVendorsName } = require('../../lib/vendors-meta');

function getBundle(bundleName, lang) {
    const modulePath = `../../../dist/ssr.bundles.${lang}`;

    if (cfg.reactRender && cfg.reactRender.isHot) {
        delete require.cache[require.resolve(modulePath)];
    }

    return require(modulePath)[bundleName]; // eslint-disable-line global-require
}
module.exports = ({ bundleName, data, language: lang, state = {}, location }) => {
    const context = {};
    const Bundle = getBundle(bundleName, lang);

    if (!Bundle) {
        throw new Error(`Bundle ${bundleName} not found`);
    }

    const bundleHtml = renderToString(
        React.createElement(
            StaticRouter,
            { context, location, basename: data.basename },
            React.createElement(
                QueryParamProvider,
                { ReactRouterRoute: Route },
                React.createElement(Bundle, { data, state }, null)
            )
        )
    );

    if (context.url) {
        return { redirectUrl: context.url };
    }

    const helmet = Helmet.rewind();

    return {
        html: getPageHtml({
            helmet,
            bundleName,
            bundleHtml,
            data,
            lang,
            state
        })
    };
};

function getPageHtml(params) {
    const { helmet, bundleName, bundleHtml, data, lang, state } = params;
    const { baseUrl, version, frozenPath } = cfg.static2;
    const bundleFilePath = `${baseUrl}${version}/${bundleName}.bundle`;
    const vendorsFilePath = `${baseUrl}${frozenPath}/${getVendorsName(lang)}`;

    const head = React.createElement(
        'head',
        null,
        helmet.title.toComponent(),
        helmet.meta.toComponent(),
        helmet.link.toComponent(),
        helmet.script.toComponent(),
        React.createElement(
            'link',
            { rel: 'stylesheet', href: `${bundleFilePath}.css` }
        )
    );

    const body = React.createElement(
        'body',
        null,
        React.createElement(
            'div',
            {
                id: 'mount',
                dangerouslySetInnerHTML: { __html: bundleHtml }
            }
        ),
        React.createElement(
            'script',
            { nonce: data.nonce, src: data.reactScript }
        ),
        React.createElement(
            'script',
            { nonce: data.nonce, src: `${vendorsFilePath}.js` }
        ),
        React.createElement(
            'script',
            { nonce: data.nonce, src: `${bundleFilePath}.${lang}.js` }
        ),
        React.createElement(
            'script',
            {
                nonce: data.nonce,
                dangerouslySetInnerHTML: {
                    __html: `Client.init(${htmlescape(data)}, ${htmlescape(state)})`
                }
            }
        )
    );

    const html = renderToStaticMarkup(
        React.createElement('html', null, head, body)
    );

    return `<!doctype html>${html}`;
}
