'use strict';

const _ = require('lodash');
const config = require('yandex-cfg');
const url = require('url');

const { buildPath } = require('../router/path');

const chinaRedirects = {
    '/agencies': '/contact/agency',
    '/about': '/yandex',
    '/story/skyscanner': '/solutions/stories/skyscanner',
    '/story/trade-ease': '/solutions',
    '/story/kuvalda': '/solutions/stories/kuvalda',
    '/story/hna': '/solutions',
    '/story/dikom': '/solutions/stories/dikom',
    '/story/wargaming': '/solutions/stories/wargaming',
    '/story/ntk-forklift': '/solutions',
    '/story/lightinthebox': '/solutions',
    '/story/quelle': '/solutions/stories/quelle',
    '/story/kayak': '/solutions/stories/kayak',
    '/story/indep': '/solutions',
    '/story/adidas': '/solutions/stories/adidas',
    '/story': '/solutions'
};

const storyRedirects = {
    '/adv/story/section': {
        url: '/adv/solutions/stories',
        hasId: true
    },
    '/adv/story/project': { url: '/adv/solutions' },
    '/adv/story/cases': { url: '/adv/solutions/cases' },
    '/adv/story/case': {
        url: '/adv/solutions/cases/:id',
        hasId: true
    },
    '/adv/story/': {
        url: '/adv/solutions/stories/:id',
        hasId: true
    },
    '/adv/story': { url: '/adv/solutions/stories' }
};

/* eslint-disable max-len */
const commonRedirects = {
    '/adv/partners/sertified-agency': '/adv/partners/certification_agency',

    // ADVDEV-993: Редиректы на курс по API Директа
    '/adv/edu/direct/direct-api/chto-takoe-api-direkta': 'https://tech.yandex.ru/direct/doc/start/intro-docpage/',
    '/adv/edu/direct/direct-api/dostup-k-api-kak-zaregistrirovat-prilozhenie-v-yandeks-oauth': 'https://tech.yandex.ru/direct/doc/start/register-docpage/',
    '/adv/edu/direct/direct-api/dostup-k-api-kak-podat-zayavku': 'https://tech.yandex.ru/direct/doc/start/access-request-docpage/',
    '/adv/edu/direct/direct-api/kak-poluchit-oauth-token': 'https://tech.yandex.ru/direct/doc/start/token-docpage/',
    '/adv/edu/direct/direct-api/kak-vklyuchit-pesochnitsu': 'https://tech.yandex.ru/direct/doc/start/sandbox-docpage/',
    '/adv/edu/direct/direct-api/kak-vypolnit-zapros-k-api': 'https://tech.yandex.ru/direct/doc/start/format-docpage/',
    '/adv/edu/direct/direct-api/kak-poluchat-dannye-cherez-api': 'https://tech.yandex.ru/direct/doc/start/get-docpage/',
    '/adv/edu/direct/direct-api/kak-vnosit-izmeneniya-cherez-api': 'https://tech.yandex.ru/direct/doc/start/modify-docpage/',
    '/adv/edu/direct/direct-api/kak-pereyti-k-realnoy-rabote': 'https://tech.yandex.ru/direct/doc/start/real-world-docpage/',
    '/adv/edu/direct/direct-api/kak-ispolzovat-api-effektivno-ogranicheniya-i-rekomendatsii': 'https://tech.yandex.ru/direct/doc/start/optimize-docpage/',
    '/adv/edu/direct/direct-api': 'https://tech.yandex.ru/direct/doc/start/index-docpage/',

    // ADVDEV-1433
    '/adv/solutions/others/hyperlocal-affinity': '/adv/solutions/analytics/hyperlocal-affinity',
    '/adv/solutions/others/auto-site-comparison': '/adv/solutions/analytics/auto-site-comparison',
    '/adv/solutions/others/expert-premium-audience': '/adv/solutions/analytics/expert-premium-audience',
    '/adv/solutions/others/pharma-links': '/adv/solutions/analytics/pharma-links',
    '/adv/solutions/others/auto-audience': '/adv/solutions/analytics/auto-audience',

    // ADVDEV-1444
    '/adv/news/novye-tseny-na-nekotorye-ploschadki-yandeksa-s-yanvarya-2008_2007-10': '/adv/news',
    '/adv/news/novye-tseny-na-nekotorye-ploschadki-yandeksa-s-yanvarya-2008_2007-12': '/adv/news',
    '/adv/news/tselevoy-zvonok-poyavilsya-v-sankt-peterburge-nizhnem-novgorode-i-samare': '/adv/news',
    '/adv/news/677': '/adv/news/679',
    '/adv/news/678': '/adv/news/679',
    '/adv/news/774': '/adv/news/novye-predlozheniya-dlya-reklamy-na-yandekse',
    '/adv/news/830': '/adv/news/na-yandeks-markete-izmenilsya-interfeys-dobavleniya-tochek-prodazh',
    '/adv/news/884': '/adv/news/mediynye-bannery-s-retargetingom-teper-i-v-rsya'
};

// ADVDEV-1204: редиректы для школ Маркета
const marketEduRedirects = {
    '/adv/edu/market/delivery-start/gde-i-kak-rabotat-s-otchetami-po-okazannym-uslugam': 'https://market.yandex.ru/partners/edu/market/delivery-start/gde-i-kak-rabotat-s-otchetami-po-okazannym-uslugam',
    '/adv/edu/market/delivery-start/kak-otgruzit-zakaz-sluzhbe-dostavki': 'https://market.yandex.ru/partners/edu/market/delivery-start/kak-otgruzit-zakaz-sluzhbe-dostavki',
    '/adv/edu/market/delivery-start/gde-i-kak-upravlyat-zakazom-posle-ego-oformleniya': 'https://market.yandex.ru/partners/edu/market/delivery-start/gde-i-kak-upravlyat-zakazom-posle-ego-oformleniya',
    '/adv/edu/market/delivery-start/kak-oformit-zakaz-na-dostavku': 'https://market.yandex.ru/partners/edu/market/delivery-start/kak-oformit-zakaz-na-dostavku',
    '/adv/edu/market/delivery-start/kak-nastroit-usloviya-dostavki-i-dostupy': 'https://market.yandex.ru/partners/edu/market/delivery-start/kak-nastroit-usloviya-dostavki-i-dostupy',
    '/adv/edu/market/delivery-start/kak-nachat-rabotu-s-yandeks-dostavkoy': 'https://market.yandex.ru/partners/edu/market/delivery-start/kak-nachat-rabotu-s-yandeks-dostavkoy',
    '/adv/edu/market/delivery-start': 'https://market.yandex.ru/partners/edu/market/delivery-start/',

    '/adv/edu/market/market-basis/otchety-na-markete': 'https://market.yandex.ru/partners/edu/market/market-basis/otchety-na-markete',
    '/adv/edu/market/market-basis/pricelabs': 'https://yandex.ru/adv/edu/market/market-basis',
    '/adv/edu/market/market-basis/kak-upravlyat-stavkami': 'https://market.yandex.ru/partners/edu/market/market-basis/kak-upravlyat-stavkami',
    '/adv/edu/market/market-basis/rabota-s-pokupatelyami-na-markete': 'https://market.yandex.ru/partners/edu/market/market-basis/rabota-s-pokupatelyami-na-markete',
    '/adv/edu/market/market-basis/tovarnye-predlozheniya-na-markete': 'https://market.yandex.ru/partners/edu/market/market-basis/tovarnye-predlozheniya-na-markete',
    '/adv/edu/market/market-basis/pokaz-na-markete': 'https://market.yandex.ru/partners/edu/market/market-basis/pokaz-na-markete',
    '/adv/edu/market/market-basis/bistroe-obnovlenie-prays-lista': 'https://market.yandex.ru/partners/edu/market/market-basis/bistroe-obnovlenie-prays-lista',
    '/adv/edu/market/market-basis/obnovlenie-prays-lista': 'https://market.yandex.ru/partners/edu/market/market-basis/obnovlenie-prays-lista',
    '/adv/edu/market/market-basis/vvedenie-ot-chego-zavisit-uspekh-magazina-na-markete': 'https://market.yandex.ru/partners/edu/market/market-basis/vvedenie-ot-chego-zavisit-uspekh-magazina-na-markete',
    '/adv/edu/market/market-basis': 'https://market.yandex.ru/partners/edu/market/market-basis',

    '/adv/edu/market/market-start/za-chto-platit-magazin-i-kak-popolnit-schet': 'https://market.yandex.ru/partners/edu/market/market-start/za-chto-platit-magazin-i-kak-popolnit-schet',
    '/adv/edu/market/market-start/kak-proyti-proverku-pri-podklyuchenii': 'https://market.yandex.ru/partners/edu/market/market-start/kak-proyti-proverku-pri-podklyuchenii',
    '/adv/edu/market/market-start/nastroyki-dostavki': 'https://market.yandex.ru/partners/edu/market/market-start/nastroyki-dostavki',
    '/adv/edu/market/market-start/registratsiya-magazina-i-startovye-nastroyki': 'https://market.yandex.ru/partners/edu/market/market-start/registratsiya-magazina-i-startovye-nastroyki',
    '/adv/edu/market/market-start/kak-pravilno-sostavit-prays-list': 'https://market.yandex.ru/partners/edu/market/market-start/kak-pravilno-sostavit-prays-list',
    '/adv/edu/market/market-start/chto-takoe-prays-list-na-markete-i-otkuda-ego-vzyat': 'https://market.yandex.ru/partners/edu/market/market-start/chto-takoe-prays-list-na-markete-i-otkuda-ego-vzyat',
    '/adv/edu/market/market-start/market-dlya-magazinov': 'https://market.yandex.ru/partners/edu/market/market-start/market-dlya-magazinov',
    '/adv/edu/market/market-start': 'https://market.yandex.ru/partners/edu/market/market-start',

    '/adv/edu/market/market-special': '/adv/edu/online/market',
    '/adv/edu/market/market-special/dostavka-marketa-podklyuchit': '/adv/edu/online/market',
    '/adv/edu/market/market-special/dostavka-marketa-zakazi': '/adv/edu/online/market'
};

const engCoursesRedirects = {
    '/adv/edu/direct-start': '/adv/edu/direct/direct-start',
    '/adv/edu/direct-start/lesson-1-yandex-direct-contextual-advertising': '/adv/edu/direct/direct-start/how-contextual-advertising-works',
    '/adv/edu/direct-start/lesson-2-how-to-select-keywords': '/adv/edu/direct/direct-start/selecting-keywords',
    '/adv/edu/direct-start/lesson-3-how-to-make-a-budget-forecast': '/adv/edu/direct/direct-start/evaluating-your-advertising-budget',
    '/adv/edu/direct-start/lesson-4-how-to-create-an-account-in-yandex-direct': '/adv/edu/direct/direct-start/creating-an-account-in-yandex-direct',
    '/adv/edu/direct-start/lesson-5-how-to-create-an-ad-campaign': '/adv/edu/direct/direct-start/creating-an-ad-campaign',
    '/adv/edu/direct-start/lesson-6-how-to-create-ad-groups': '/adv/edu/direct/direct-start/create-an-ad-group',
    '/adv/edu/direct-start/lesson-7-how-to-pass-moderation': '/adv/edu/direct/direct-start/passing-moderation',
    '/adv/edu/direct-start/lesson-8-how-to-pay-for-and-launch-an-advertising-campaign': '/adv/edu/direct/direct-start/paying-for-and-launching-your-ad-campaign'
};

const partnersRequestRedirects = {
    '/adv/partners/request': '/adv/agency-request'
};

const analyticsNowRedirects = {
    '/adv/analytics-now/adapting-ad-campaigns-for-the-stay-at-home-period': '/adv/news/adapting-ad-campaigns-for-the-stay-at-home-period',
    '/adv/analytics-now': '/adv/news'
};


// ADVDEV-1798: Четыре (по три) редиректа со страниц новостей в практикумы
const practicumsRedirectsTlds = ['ru', 'kz', 'by'];
const practicumsRedirects = {
    '/adv/news/tonkosti-retargetinga-chek-list-po-nastroyke': '/adv/solutions/practicums/tonkosti-retargetinga-chek-list-po-nastroyke',
    '/adv/news/chek-list-po-nastroyke-dinamicheskikh-obyavleniy': '/adv/solutions/practicums/chek-list-po-nastroyke-dinamicheskikh-obyavleniy',
    '/adv/news/kak-legko-sozdat-reklamnyy-videorolik-5-vozmozhnostey-videokonstruktora-v-direkte': '/adv/solutions/practicums/kak-legko-sozdat-reklamnyy-videorolik',
    '/adv/news/chek-list-po-nastroyke-okhvatnoy-videoreklamy-v-direkte': '/adv/solutions/practicums/chek-list-po-nastroyke-okhvatnoy-videoreklamy-v-direkte',
    '/adv/news/smart-bannery-po-umnomu-chek-list-po-nastroyke': '/adv/solutions/practicums/smart-bannery-chek-list-po-nastroyke'
};

const newsRedirects = {
    // ADVDEV-1885
    '/adv/news/obuchaemye-segmenty-v-yandeks-auditoriyakh-bolshe-konversiy-v-slozhnykh-tematikakh': {
        to: '/adv/news/kak-poluchat-bolshe-kachestvennykh-konversiy-v-direkte-s-obuchaemymi-segmentami-yandeks-auditoriy',
        tlds: ['ru']
    }
};

/* eslint-enable max-len */

const handlers = [
    function domainsNotExistsRedirect(req) {
        const domains = config.app.domains.split(',');

        if (domains.indexOf(req.tld) < 0) {
            const { host } = config;
            const [domain] = config.regionsOrder;
            const path = config.router.adv;

            return { code: 302, url: `${host}${domain}${path}` };
        }
    },

    function skipTrailingSlash(req) {
        const parsedUrl = url.parse(req.originalUrl);

        if (parsedUrl.pathname !== '/adv/' && /.\/$/.test(parsedUrl.pathname)) {
            parsedUrl.pathname = parsedUrl.pathname.slice(0, -1);

            return { code: 301, url: url.format(parsedUrl) };
        }
    },

    function wwwRedirect(req) {
        if (req.currentUrl.match(/^https?:\/\/www\./)) {
            const redirectUrl = req.currentUrl.replace(/www\./, '');

            return { code: 301, url: redirectUrl };
        }
    },

    function cnRedirects(req) {
        if (req.tld !== 'cn') {
            return;
        }

        const currentUrl = req.originalUrl.replace(/^\/adv/, '');

        for (const path of Object.keys(chinaRedirects)) {
            if (currentUrl === path) {
                return { code: 301, url: chinaRedirects[path] };
            }
        }
    },

    function storyRedirectsHandler(req) {
        for (const path of Object.keys(storyRedirects)) {
            if (!req.originalUrl.startsWith(path)) {
                continue;
            }

            const redirectObj = storyRedirects[path];
            const id = redirectObj.hasId ? req.originalUrl.split('/').pop() : '';
            const redirectPath = redirectObj.url.replace(/:id/g, id);

            if (_.includes(config.story.domains, req.tld)) {
                return { code: 301, url: redirectPath };
            }

            return { code: 301, url: buildPath(req.currentUrl, '/') };
        }
    },

    function commonRedirectsHandler(req) {
        for (const path of Object.keys(commonRedirects)) {
            if (req.originalUrl.startsWith(path)) {
                return { code: 301, url: commonRedirects[path] };
            }
        }
    },

    function engCoursesRedirectsHandler(req) {
        if (req.tld !== 'com') {
            return;
        }

        for (const path of Object.keys(engCoursesRedirects)) {
            if (req.originalUrl.startsWith(path)) {
                return { code: 301, url: engCoursesRedirects[path] };
            }
        }
    },

    function marketEduRedirectsHandler(req) {
        if (req.tld !== 'ru') {
            return;
        }

        for (const path of Object.keys(marketEduRedirects)) {
            if (req.originalUrl.startsWith(path)) {
                return { code: 301, url: marketEduRedirects[path] };
            }
        }
    },

    function partnersRequestRedirectsHandler(req) {
        if (!_.includes(['ru', 'by', 'kz'], req.tld)) {
            return;
        }

        for (const path of Object.keys(partnersRequestRedirects)) {
            if (req.originalUrl.startsWith(path)) {
                return { code: 301, url: partnersRequestRedirects[path] };
            }
        }
    },

    function analyticsNowRedirectsHandler(req) {
        if (req.tld !== 'com') {
            return;
        }

        for (const path of Object.keys(analyticsNowRedirects)) {
            if (req.originalUrl.startsWith(path)) {
                return { code: 302, url: analyticsNowRedirects[path] };
            }
        }
    },

    function practicumsRedirectsHandler(req) {
        if (!practicumsRedirectsTlds.includes(req.tld)) {
            return;
        }

        for (const path of Object.keys(practicumsRedirects)) {
            if (req.originalUrl.startsWith(path)) {
                return { code: 301, url: `${config.host}ru${practicumsRedirects[path]}` };
            }
        }
    },

    function newsRedirectsHandler(req) {
        for (const path of Object.keys(newsRedirects)) {
            if (req.originalUrl.startsWith(path)) {
                return newsRedirects[path].tlds.includes(req.tld) ?
                    { code: 301, url: newsRedirects[path].to } :
                    {};
            }
        }
    }

];

module.exports = (req, res, next) => {
    try {
        req.currentUrl = decodeURIComponent(url.format({
            protocol: 'https',
            host: req.headers.host,
            pathname: req.originalUrl
        }));
    } catch (e) {
        res.sendStatus(404);

        return;
    }

    for (const handler of handlers) {
        const redirect = handler(req) || {};

        if (redirect.code && redirect.url) {
            return res.redirect(redirect.code, redirect.url);
        }
    }

    next();
};
