'use strict';

const _ = require('lodash');

const Base = require('./base');
const { decodeAttributes, dateMonthYearSorter } = require('../../lib/helper');

/**
 * Набор функций для сортировки значений в фильтрах
 * @type {Object}
 */
const sorters = {
    // Лексикографический порядок (по умолчанию)
    text: 'name',

    // Числовой формат
    number: entity => parseFloat(_.get(entity.name.match(/^.*?(\d+([.,]\d+)?).*$/), 1, '')),

    // Дата в формате "Месяц ГГГГ"
    dateMonthYear: dateMonthYearSorter
};

/**
 * Абстрактный класс для моделей, фильтрующих коллекции
 */
class Filterable extends Base {
    constructor(req, attributes = {}) {
        super(req, attributes);

        const filterOptions = _.omit(this._attributes, ['filters', 'startIdx']);

        this.values = decodeAttributes(filterOptions);
        this.slugs = _.keys(this.values);
    }

    /**
     * Фильтрует набор сущностей по параметрам
     * @param {{ filters: { slug: String, values: String }[] }[]} collection
     * @returns {LodashWrapper<Object[]>}
     * @protected
     */
    _filterCollection(collection) {
        return _(collection).filter(this._filterEntity.bind(this));
    }

    /**
     * Возвращает данные для отображения фильтров
     * @param {Object[]} pageFilters
     * @param {Object[]} collection
     * @returns {Object}
     * @protected
     */
    _getFiltersData(pageFilters, collection) {
        return pageFilters.map(pageFilter => {
            const values = this._getFilterValues(pageFilter, collection);

            return _.assign({ values }, pageFilter);
        });
    }

    /**
     * Фильтр для элемента коллекции.
     * Пробегаем по всем фильтрам сущности. Проверяем, что:
     *      - либо слаг фильтра не задан в параметрах для фильтрации
     *      - либо значения фильтра содержатся в параметрах для фильтрации
     * @param {{ filters: { slug: String, values: String }[] }} entity
     * @returns {Boolean}
     * @private
     */
    _filterEntity(entity) {
        const filters = _.get(entity, 'filters', []);

        const areFilterValuesIntersected = filters
            .every(this._hasFilterValues, this);

        const hasOptionsForEachFilter = this.slugs
            .every(slug => _.find(filters, { slug }));

        return areFilterValuesIntersected && hasOptionsForEachFilter;
    }

    /**
     * Проверяет, что значения фильтров содержатся в параметрах для фильтрации
     * @param {{ slug: String, values: String }} filter
     * @returns {Boolean}
     * @private
     */
    _hasFilterValues(filter) {
        const slugValues = this.values[filter.slug];

        return _.isEmpty(slugValues) || !_(filter.values)
            .intersection(slugValues)
            .isEmpty();
    }

    /**
     * Возвращает данные для отображения фильтра
     * @param {{ slug: String, type: String }} pageFilter
     * @param {Object[]} collection
     * @returns {Array}
     * @private
     */
    _getFilterValues(pageFilter, collection) {
        const { slug } = pageFilter;
        const values = _.get(this.values, slug, []);
        let sorter = _.get(sorters, pageFilter.type, sorters.text);

        if (pageFilter.type === 'dateMonthYear') {
            sorter = sorter.bind(this._req, 'name', 'desc');
        }

        return _(collection)
            .flatMap('filters')
            .filter({ slug })
            .flatMap('values')
            .uniq()
            .map(name => {
                const checked = _.includes(values, name);

                return { name, checked };
            })
            .sortBy(sorter)
            .value();
    }
}

module.exports = Filterable;
