'use strict';

const _ = require('lodash');
const Promise = require('bluebird');
const qs = require('qs');
const url = require('url');

const Filterable = require('./abstractions/filterable');
const { decodeAttributes, getBunkerNode } = require('../lib/helper');
const shuffle = require('../lib/shuffle');

const CHUNK_LENGTH = 10;

/**
 * Модель агентства
 */
class Agencies extends Filterable {
    constructor(req, attributes = {}) {
        super(req, attributes);

        const attrs = _.omit(attributes, [
            'filters',
            'startIdx',
            'countries',
            'services',
            'cities',
            'fetchAll',
            'site',
            'permutation'
        ]);

        this.values = decodeAttributes(attrs);
        this.slugs = _.keys(this.values);
    }

    /**
     * Возвращает обязательные параметры для фильтрации: страна, город, сервис
     * @returns {Object}
     */
    get defaultFilters() {
        const countryFilter = this._req.locals.countryId;
        const filters = _.pick(this._attributes, ['countries', 'cities', 'services']);

        if (countryFilter) {
            filters.countries = [countryFilter];
        }

        return filters;
    }

    getRegionsData() {
        return getBunkerNode(this._req.tld, this._bunker.sources, {
            path: ['contact', 'agencies', 'regions']
        });
    }

    fetchSearch() {
        /* eslint-disable camelcase */
        const currentCountry = this._req.locals.countryId;
        const root = getBunkerNode(this._req.tld, this._bunker.sources, {
            path: ['contact', 'agencies']
        });
        const agencies = _.pick(root, [
            'title',
            'description',
            'citiesByCountries',
            'services',
            'filters'
        ]);

        if (!agencies) {
            return {};
        }

        if (currentCountry) {
            agencies.citiesByCountries = _.filter(agencies.citiesByCountries, {
                geo_id: currentCountry
            });
        }

        return agencies;
    }

    /**
     * Получение списка агенств
     * @returns {Promise}
     */
    fetch() {
        return Promise.resolve(this._bunker.agencies)
            .bind(this)
            .then(this._parseData)
            .then(this._filterAgencies);
    }

    _parseData(agencies) {
        const filters = this.defaultFilters;

        // собираем агенства со всех стран
        return _(agencies)
            .values()
            .map(_.values)
            .flatten()
            .filter(agency => !agency.disabled && _.isObject(agency))
            // выбираем язык
            .map(agency => {
                // req.bunker шарится между запросами, так как берется из кеша
                agency = _.cloneDeep(agency);
                agency.name = this._getFieldByLang(agency.name);
                agency.description = this._getFieldByLang(agency.description);
                agency.addresses = agency.addresses.map(address => {
                    address.address = this._getFieldByLang(address.address);

                    return address;
                });

                const query = _.extend({ url: agency.site }, filters);

                agency.url = url.format({
                    pathname: this._buildPath('contact', 'agency', 'partner'),
                    search: qs.stringify(query, { arrayFormat: 'brackets' })
                });

                agency.logo = agency.logo || 'https://yastatic.net/s3/vda/i/images/agency.png';

                return agency;
            })
            .value();
    }

    /**
     * Получает поле по локали
     * @param {Object} field
     * @returns {*}
     * @private
     */
    _getFieldByLang(field) {
        const langId = this._req.language;
        const lang = ['en', 'tr'].indexOf(langId) > -1 ? 'native' : 'ru';

        return field[lang];
    }

    /**
     * Фильтрация агенств по параметрам
     * @param {Array} agencies
     * @returns {Object}
     * @private
     */
    _filterAgencies(agencies) {
        const startIdx = this._attributes.startIdx || 0;
        const lastIdx = startIdx + CHUNK_LENGTH;
        const filtersData = _.get(this.fetchSearch(), 'filters', {});
        const shuffleResult = shuffle(agencies, this._attributes.permutation);
        const localAgencies = this._filterByCountry(shuffleResult.collection);
        const defaultFiltered = this._filterByDefault(localAgencies);
        const filteredAgencies = this._filterCollection(defaultFiltered).value();

        return {
            list: this._attributes.fetchAll ? filteredAgencies :
                filteredAgencies.slice(startIdx, lastIdx),
            more: lastIdx < filteredAgencies.length,
            lastIdx,
            filters: this.defaultFilters,
            customFilters: this._getFiltersData(filtersData, localAgencies),
            permutation: shuffleResult.permutation
        };
    }

    /**
     * Фильтрует агентства по странам
     * @param {Object []} agencies
     * @returns {Object []}
     * @private
     */
    _filterByCountry(agencies) {
        const countries = _.get(this.defaultFilters, 'countries', []);
        const hasCountries = countries && countries.length;

        return agencies.filter(agency => {
            return !hasCountries || agency.addresses
                .some(address => _.includes(countries, address.country));
        });
    }

    /**
     * Функция для фильтрации агентств по городам и сервисам
     * @param {Object []} agencies
     * @returns {Object []}
     */
    _filterByDefault(agencies) {
        const { cities, services } = this.defaultFilters || {};

        /* eslint-disable complexity */
        return agencies.filter(agency => {
            const hasCities = cities && cities.length;
            const containsCity = !hasCities || _.includes(cities, agency.addresses[0].city);
            const hasServices = services && services.length;
            const containsService = !hasServices ||
                _.some(services, service => _.get(agency.certificates, service));

            return containsCity && containsService;
        });
        /* eslint-enable complexity */
    }
}

module.exports = Agencies;
