'use strict';

const Agencies = require('./agencies');
const Promise = require('bluebird');
const _ = require('lodash');
const qs = require('qs');
const url = require('url');
const getMapsApi = require('../lib/getMapsApi');

class Agency extends Agencies {

    /**
     * Возвращает агенство
     * @returns {Promise}
     */
    get() {
        this._attributes = _.assign(this._attributes, { fetchAll: true });

        return this
            .fetch()
            .then(this._getAgency)
            .then(this._addAgencyUrls)
            .then(this._addAgencyEmails)
            .then(this._getCertificates)
            .then(this._addMapsInfo);
    }

    _getAgency(data) {
        const agencies = data.list;
        const agency = _.find(agencies, { site: this._attributes.site });

        if (_.isEmpty(agency)) {
            return Promise.reject({
                internalCode: '404_ANF',
                message: 'agency was not found'
            });
        }

        const currentAgencyIndex = agencies.indexOf(agency);

        return {
            agency,
            prevAgencyUrl: agencies[(currentAgencyIndex || agencies.length) - 1].url,
            nextAgencyUrl: agencies[(currentAgencyIndex + 1) % agencies.length].url
        };
    }

    _addAgencyUrls(data) {
        const params = _.pick(this._attributes, ['countries', 'cities', 'services']);

        return _.extend(data, {
            agenciesUrl: url.format({
                pathname: this._buildPath('contact', 'agency'),
                search: qs.stringify(params, { arrayFormat: 'brackets' })
            })
        });
    }

    _addAgencyEmails(data) {
        data.agency.emails = _(data.agency.addresses)
            .map(address => address.email)
            .uniq()
            .value();

        return data;
    }

    _getCertificates(data) {
        return data;
    }

    _addMapsInfo(data) {
        data.mapsApi = getMapsApi(this._req.tld, data.agency.addresses.map(addr => addr.map));

        return data;
    }
}

module.exports = Agency;
