'use strict';

const _ = require('lodash');
const EduBase = require('./edu-base');
const Promise = require('bluebird');

/**
 * Получает всех экспертов
 * @param {Object} expertsData
 * @returns {Object []}
 */
function getAllExperts(expertsData) {
    return _(expertsData)
        .values()
        .value();
}

/**
 * Выбирает экспертов данного события
 * @param {Object} event
 * @param {Object []} experts
 * @returns {Object []}
 */
function getEventExperts(event, experts) {
    return event.experts && _(event.experts)
        .map(expertSlug => {
            const expert = _(experts)
                .find(entity => entity.slug === expertSlug);

            if (!expert) {
                return;
            }

            return expert;
        })
        .compact()
        .value();
}

class Event extends EduBase {
    fetch() {
        const { tld } = this._req;

        return Promise.resolve({
            events: this._bunker.events[tld],
            experts: this._bunker.experts[tld]
        })
            .bind(this)
            .then(this._parseData);
    }

    _parseData(data) {
        const { eventSlug } = this._attributes;
        const event = this._getEvent(data.events, eventSlug);

        if (!event) {
            return;
        }

        const experts = getAllExperts(data.experts);
        const eventExperts = getEventExperts(event, experts);
        const expertSlugToUrl = eventExperts && eventExperts.reduce((current, expert) => {
            current[expert.slug] = this._buildPath('edu', 'experts', expert.slug);

            return current;
        }, {});

        return {
            event,
            experts: eventExperts,
            expertSlugToUrl
        };
    }

    /**
     * Возвращает запрашиваемое событие по слагу, если оно не отключено
     * @param {Object} eventData
     * @param {String} eventSlug
     * @returns {Object}
     */
    _getEvent(eventData, eventSlug) {
        return _(eventData)
            .values()
            .find(event => this._isActualEvent(event) && event.slug === eventSlug);
    }
}

module.exports = Event;
