'use strict';

const _ = require('lodash');
const EduBase = require('./edu-base');
const Promise = require('bluebird');
const helper = require('../../lib/helper');

/**
 * Возвращает массив, содержащий события, используя функцию преобразования
 * @param {Object} data
 * @param {Function} func
 * @returns {Object []}
 */
function mapEventsDistinct(data, func) {
    return _(data.events)
        .map(func)
        .flatten()
        .uniq()
        .compact()
        .value();
}

/**
 * Получает диапазон месяцев, в которые проходит данное событие
 * @param {Object} event
 * @returns {Number []}
 */
function getAllMonths(event) {
    const startMonth = event.start && helper.getMonth(event.start.date);
    const months = [startMonth];

    if (event.end) {
        const endMonth = helper.getMonth(event.end.date);

        for (let number = startMonth + 1; number <= endMonth; number += 1) {
            months.push(number);
        }
    }

    return months;
}

/**
 * Проверяет, что город данного события содержится в городах для фильтрации
 * @param {Object} filters
 * @param {String} langId
 * @param {Object} event
 * @returns {Boolean}
 */
function belongsToCities(filters, langId, event) {
    return !filters.cities.length || !event.address ||
        _.includes(filters.cities, (event.address[langId] || event.address.native).city);
}

/**
 * Проверяет, что тематика данного события содержится в тематиках для фильтрации
 * @param {Object} filters
 * @param {Object} event
 * @returns {Boolean}
 */
function belongsToThemes(filters, event) {
    return !filters.themes.length || _.includes(filters.themes, event.theme);
}

/**
 * Проверяет, что месяц данного события содержится в месяцах для фильтрации
 * @param {Object} filters
 * @param {Object} event
 * @returns {Boolean}
 */
function belongsToMonths(filters, event) {
    if (filters.months.length === 0) {
        return true;
    }

    const startMonth = helper.getMonth(event.start && event.start.date);

    if (event.end && event.end.date) {
        const endMonth = helper.getMonth(event.end.date);

        return _.some(filters.months, month => {
            const monthNumber = Number(month);

            return _.inRange(monthNumber, startMonth, endMonth + 1);
        });
    }

    return _.includes(filters.months, startMonth.toString());
}

class Events extends EduBase {
    fetchPage() {
        const eventsPage = helper.getBunkerNode(this._req.tld,
            this._bunker.sources,
            {
                path: ['edu', 'events']
            });
        const events = _.pick(eventsPage, ['title', 'description', 'additionalText', 'regions']);

        if (!events) {
            return {};
        }

        return events;
    }

    fetch() {
        const { tld } = this._req;

        return Promise.resolve({
            events: this._bunker.events[tld],
            experts: this._bunker.experts[tld]
        })
            .bind(this)
            .then(this._parseData)
            .then(this._getFiltersData)
            .then(this._filterEvents);
    }

    _parseData(eventsData) {
        const parsedEvents = _(eventsData.events)
            .values()
            .filter(this._isActualEvent)
            .sortBy(event => helper.getDate(_.get(event, 'start.date', '')))
            .value();

        const expertSlugToName = _(eventsData.experts)
            .values()
            .reduce((current, expert) => {
                current[expert.slug] = expert.name;

                return current;
            }, {});

        const eventSlugToUrl = parsedEvents.reduce((current, event) => {
            current[event.slug] = event.skipAnnouncement ?
                event.registerButton.url : this._buildPath('edu', 'events', event.slug);

            return current;
        }, {});

        return {
            events: parsedEvents,
            eventSlugToUrl,
            expertSlugToName
        };
    }

    _getFiltersData(data) {
        const langId = this._req.language;
        const themes = mapEventsDistinct(data, event => event.theme)
            .sort((thisEvent, thatEvent) => {
                if (thisEvent > thatEvent) {
                    return 1;
                }
                if (thisEvent < thatEvent) {
                    return -1;
                }

                return 0;
            });
        const months = mapEventsDistinct(data, getAllMonths)
            .sort((thisMonth, thatMonth) => thisMonth - thatMonth);

        data.eventsSearch = {
            cities: mapEventsDistinct(data, event => event.address &&
                (event.address[langId] || event.address.native).city),
            themes,
            months
        };

        return data;
    }

    _filterEvents(data) {
        const langId = this._req.language;
        const filters = _.pick(this._attributes, ['cities', 'themes', 'months']);

        if (filters && Object.keys(filters).length) {
            data.events = data.events.filter(event => {
                return belongsToCities(filters, langId, event) &&
                    belongsToThemes(filters, event) &&
                    belongsToMonths(filters, event);
            });
        }

        return data;
    }
}

module.exports = Events;
