'use strict';

const _ = require('lodash');
const EduBase = require('./edu-base');
const Promise = require('bluebird');

/**
 * Возвращает эксперта с данным слагом, если он не отключен
 * @param {Object} expertData
 * @param {String} expertSlug
 * @returns {Object}
 */
function getExpert(expertData, expertSlug) {
    return _(expertData)
        .values()
        .find(expert => _.isObject(expert) && expert.enabled &&
            !expert.isYandexSpeaker && expert.slug === expertSlug);
}

/**
 * Получает массив из всех событий
 * @param {Object} eventsData
 * @returns {Object []}
 */
function getAllEvents(eventsData) {
    return _(eventsData)
        .values()
        .value();
}

class Expert extends EduBase {
    fetch() {
        const { tld } = this._req;

        return Promise.resolve({
            experts: this._bunker.experts[tld],
            events: this._bunker.events[tld]
        })
            .bind(this)
            .then(this._parseData);
    }

    _parseData(data) {
        const { expertSlug } = this._attributes;
        const expert = getExpert(data.experts, expertSlug);

        if (!expert) {
            return;
        }

        const events = getAllEvents(data.events);
        const expertEvents = this._getExpertEvents(expertSlug, events);
        const eventSlugToUrl = expertEvents && expertEvents.reduce((current, event) => {
            current[event.slug] = this._buildPath('edu', 'events', event.slug);

            return current;
        }, {});

        const expertSlugToName = _(data.experts)
            .values()
            .reduce((current, entity) => {
                current[entity.slug] = entity.name;

                return current;
            }, {});

        return {
            expert,
            events: expertEvents,
            eventSlugToUrl,
            expertSlugToName
        };
    }

    /**
     * Возвращает события, в которых указан данный эксперт
     * @param {String} expertSlug
     * @param {Object []} events
     * @returns {Object []}
     */
    _getExpertEvents(expertSlug, events) {
        return expertSlug && _(events)
            .filter(event => this._isActualEvent(event) && _.includes(event.experts, expertSlug))
            .value();
    }
}

module.exports = Expert;
