'use strict';

const _ = require('lodash');
const Base = require('../abstractions/base');
const helper = require('../../lib/helper');
const Promise = require('bluebird');

/**
 * Возвращает массив, содержащий экспертов, используя функцию преобразования
 * @param {Object} data
 * @param {Function} func
 * @returns {Object []}
 */
function mapExpertsDistinct(data, func) {
    return _(data.experts)
        .map(func)
        .flatten()
        .uniq()
        .compact()
        .value();
}

/**
 * Получает значения, по которым ведётся фильтрация
 * @param {Object} data
 * @returns {Object}
 */
function getFiltersData(data) {
    data.expertsSearch = {
        cities: mapExpertsDistinct(data, expert => expert.city),
        specializations: mapExpertsDistinct(data, expert => expert.specializations)
    };

    return data;
}

/**
 * Проверяет, что хотя бы один из элементов принадлежит массиву
 * @param {Array} items
 * @param {Array} array
 * @returns {Boolean}
 */
function someBelongsToArray(items, array) {
    return items.some(item => _.includes(array, item));
}

/**
 * Проверяет, что город данного эксперта содержится в городах для фильтрации
 * @param {Object} filters
 * @param {Object} expert
 * @returns {Boolean}
 */
function belongsToCities(filters, expert) {
    return !filters.cities.length || _.includes(filters.cities, expert.city);
}

/**
 * Проверяет, что специализация данного эксперта содержится в специализациях для фильтрации
 * @param {Object} filters
 * @param {Object} expert
 * @returns {Boolean}
 */
function belongsToSpecializations(filters, expert) {
    return !filters.specializations.length ||
        someBelongsToArray(expert.specializations, filters.specializations);
}

/**
 * Проверяет, что формат работы данного эксперта содержится в форматах работы для фильтрации
 * @param {Object} filters
 * @param {Object} expert
 * @returns {Boolean}
 */
function belongsToWorkFormats(filters, expert) {
    return !filters.workFormats.length ||
        someBelongsToArray(expert.workFormats, filters.workFormats);
}

class Experts extends Base {
    fetchPage() {
        const expertsPage = helper.getBunkerNode(this._req.tld,
            this._bunker.sources,
            {
                path: ['edu', 'experts']
            });
        const experts = _.pick(expertsPage, [
            'title',
            'regions',
            'imageUrl',
            'description',
            'additionalText'
        ]);

        if (!experts) {
            return {};
        }

        return experts;
    }

    fetch() {
        const { tld } = this._req;

        return Promise.resolve(this._bunker.experts[tld])
            .bind(this)
            .then(this._parseData)
            .then(getFiltersData)
            .then(this._filterExperts);
    }

    _parseData(experts) {
        const parsedExperts = _(experts)
            .values()
            .filter(expert => _.isObject(expert) && expert.enabled && !expert.isYandexSpeaker)
            .sortBy(helper.getSurname)
            .value();

        const expertSlugToUrl = parsedExperts.reduce((current, expert) => {
            current[expert.slug] = this._buildPath('edu', 'experts', expert.slug);

            return current;
        }, {});

        return {
            experts: parsedExperts,
            expertSlugToUrl
        };
    }

    _filterExperts(data) {
        const filters = _.pick(this._attributes, ['cities', 'specializations', 'workFormats']);

        if (filters && Object.keys(filters).length) {
            data.experts = data.experts.filter(expert => {
                return belongsToCities(filters, expert) &&
                    belongsToSpecializations(filters, expert) &&
                    belongsToWorkFormats(filters, expert);
            });
        }

        return data;
    }
}

module.exports = Experts;
