'use strict';

const _ = require('lodash');
const Promise = require('bluebird');

const helper = require('../../../lib/helper');

const Base = require('../../abstractions/base');

const filterEnabledNodes = node => _.isObject(node) && node.enabled;

class Courses extends Base {
    get onlineNode() {
        return helper.getBunkerNode(this._req.tld,
            this._req.bunker.sources,
            {
                path: 'edu.online'
            });
    }

    fetch() {
        return Promise.resolve(this.onlineNode)
            .then(this._parseCourse.bind(this))
            .then(this._parseLesson.bind(this));
    }

    _parseCourse(data) {
        const { service, course } = this._attributes;

        const serviceNode = _(data)
            .values()
            .find(node => _.get(node, 'slug') === service);

        const courseNode = serviceNode && _(serviceNode.courses)
            .values()
            .find(node => _.isObject(node) && node.enabled && node.slug === course);

        const lessons = courseNode && _(courseNode)
            .values()
            .filter(filterEnabledNodes)
            .orderBy('order')
            .value();

        const courseLink = this._buildPath('edu', service, course);
        const section = _.get(serviceNode, 'slug');

        return { storage: courseNode, lessons, courseLink, section };
    }

    _parseLesson(data) {
        const { service, course, lesson } = this._attributes;

        if (!lesson) {
            return data;
        }

        const lessonNode = _(data.lessons)
            .find(node => node.slug === lesson);

        const prevLesson = _(data.lessons)
            .find(node => lessonNode && node.order === lessonNode.order - 1);

        const nextLesson = _(data.lessons)
            .find(node => lessonNode && node.order === lessonNode.order + 1);

        const blocks = _(lessonNode)
            .values()
            .filter(filterEnabledNodes)
            .map(this._extendLesson.bind(this))
            .orderBy('order')
            .value();

        const lessonsLink = this._buildPath('edu', service, course);

        return {
            section: data.section,
            storage: lessonNode,
            lessonsLink,
            prevLesson,
            nextLesson,
            blocks
        };
    }

    // FIXME https://st.yandex-team.ru/ADVDEV-853
    _extendLesson(lessonNode) {
        switch (lessonNode.type) {
            case 'pins':
                return this._extendLessonWithPins(lessonNode);

            case 'rtb-block':
                return this._extendLessonWithRTBBlocks(lessonNode);

            default:
                return lessonNode;
        }
    }

    _extendLessonWithPins(lessonNode) {
        const pinsData = helper.getBunkerNode(this._req.tld,
            this._req.bunker.sources,
            {
                path: ['images-pins', lessonNode.pins]
            });

        return _.assign({}, lessonNode, { pinsData });
    }

    _extendLessonWithRTBBlocks(lessonNode) {
        const rtbBlock = helper.getBunkerNode(this._req.tld,
            this._req.bunker.sources,
            {
                path: ['rtb-blocks', lessonNode.rtbBlock]
            }
        );

        return _.assign({}, lessonNode, { rtbBlock });
    }

}

module.exports = Courses;
