'use strict';

const _ = require('lodash');
const Promise = require('bluebird');

const Base = require('../../abstractions/base');
const helper = require('../../../lib/helper');

/**
 * Находит сервис и урок по переданным слагам
 * @param {Object} onlineEduNode
 * @param {String} serviceSlug
 * @param {String} lessonSlug
 * @returns {{service, lesson: Object}}
 */
function getServiceAndLesson(onlineEduNode, serviceSlug, lessonSlug) {
    const service = _(onlineEduNode)
        .values()
        .find(item => _.isObject(item) && item.slug === serviceSlug &&
            (!lessonSlug || item.isService));

    const lesson = lessonSlug && service && _(service)
        .values()
        .find(item => _.isObject(item) && item.slug === lessonSlug);

    return { service, lesson };
}

class OnlineService extends Base {
    fetch() {
        const { service, lesson } = this._attributes;

        if (!service) {
            return Promise.resolve({ });
        }

        const onlineEduNode = helper.getBunkerNode(this._req.tld,
            this._req.bunker.sources,
            {
                path: 'edu.online'
            });

        return Promise
            .resolve(getServiceAndLesson(onlineEduNode, service, lesson))
            .then(this._parseData.bind(this));
    }

    _parseData(data) {
        const { service, lesson } = data;

        if (!service) {
            return;
        }

        const lessonToParse = lesson || (!service.isService && service);
        let page = 'online-edu-service';
        let serviceData = service;

        if (lessonToParse) {
            page = 'online-edu-lesson';

            serviceData = _.assign({
                levels: helper.getPageLevels(lessonToParse)
            }, lessonToParse);
        }

        return {
            page,
            section: service.slug,
            data: serviceData
        };
    }
}

module.exports = OnlineService;
