'use strict';

const _ = require('lodash');
const Promise = require('bluebird');

const { avatarsBase, certificatesIcons } = require('yandex-cfg');
const Filterable = require('./abstractions/filterable');
const { getSurname, getBunkerNode, joinArraysBySlug } = require('../lib/helper');

class Freelancers extends Filterable {

    /**
     * Возвращает отфильтроавнный список фрилансеров.
     * Принимает key-value атрибуты для фильтрации.
     * @returns {Promise<Object|null>}
     */
    fetchList() {
        const page = this._getFreelancersPage();
        const freelancers = this._getFreelancersQueryable();

        if (!page) {
            return Promise.resolve(null);
        }

        return Promise.resolve({ page, freelancers })
            .then(this._parseList.bind(this));
    }

    /**
     * Возвращает фрилансера по слагу.
     *     slug - слаг фрилансера
     * @returns {Promise<Object|null>}
     */
    fetch() {
        const { slug } = this._attributes;
        const freelancer = this._getFreelancersQueryable()
            .find(freelancerData => freelancerData.slug === slug);

        if (!freelancer) {
            return Promise.resolve(null);
        }

        return Promise.resolve(freelancer)
            .then(this._parseFreelancer.bind(this));
    }

    /**
     * Добавляет опции для фильтров и фильтрует фрилансеров.
     * @param {{ page: Object, freelancers: Object [] }} data
     * @returns {Object}
     * @private
     */
    _parseList(data) {
        const page = _.get(data, 'page', {});
        const filters = this._getFiltersData(page.filters, data.freelancers);
        const freelancers = this._filterCollection(data.freelancers)
            .map(freelancer => _.assign(freelancer, {
                bigPhoto: freelancer.image,
                filters: joinArraysBySlug(freelancer.filters, page.filters)
            }))
            .sortBy(getSurname)
            .value();

        const expertSlugToUrl = freelancers.reduce((acc, freelancer) => {
            acc[freelancer.slug] = this._buildPath('contact', 'freelancers', freelancer.slug);

            return acc;
        }, {});

        return _.merge({ filters, freelancers, expertSlugToUrl }, page);
    }

    /**
     * Добавляет названия для фильтров и изображения для сертификатов.
     * @param {{ filters: Object [] }} data
     * @returns {Object}
     * @private
     */
    _parseFreelancer(data) {
        const page = this._getFreelancersPage();
        const { contactForm } = page;

        const filters = joinArraysBySlug(data.filters, page.filters);

        const certificates = _.get(data, 'certificates', [])
            .map(certificate => {
                const image = _.get(certificatesIcons, certificate.type);

                return _.assign({ image: image && `${avatarsBase}${image}/orig` }, certificate);
            });

        return _.merge({ filters, certificates, contactForm }, data);
    }

    /**
     * Возвращает LodashWrapper над списком фрилансеров
     * @returns {LodashWrapper}
     * @private
     */
    _getFreelancersQueryable() {
        const freelancersNode = getBunkerNode(this._req.tld, this._bunker.freelancers || {}, {
            useRequestedTldOnly: true
        });

        return _(freelancersNode)
            .values()
            .filter(freelancer => freelancer.enabled);
    }

    /**
     * Получает страницу списка фрилансеров
     * @returns {Object}
     * @private
     */
    _getFreelancersPage() {
        const freelancersPage = getBunkerNode(this._req.tld, this._bunker.sources, {
            path: 'freelancers'
        });
        const freelancers = _.pick(freelancersPage, [
            'title',
            'image',
            'description',
            'additionalText',
            'disclaimer',
            'regions',
            'filters',
            'contactForm'
        ]);

        return freelancers || {};
    }
}

module.exports = Freelancers;
