const _ = require('lodash');
const Promise = require('bluebird');

const { getBunkerNode, dateMonthYearSorter } = require('../lib/helper');

const Filterable = require('./abstractions/filterable');

const filterSlugs = ['date', 'theme'];

class JournalsModel extends Filterable {
    constructor(req, attributes = {}) {
        super(req, attributes);

        const path = 'journals';

        this._chunkLength = 10;
        this.bunkerNode = getBunkerNode(this._req.tld, this._req.bunker.sources, { path });
        this.journals = _(this.bunkerNode)
            .values()
            .filter('enabled')
            .map(this._assignFilters)
            .value();
    }

    /**
     * Получить актуальный журнал
     * @returns {Promise<Object?>}
     */
    get actual() {
        const actual = _.find(this.journals, 'actual');

        return Promise.resolve(actual);
    }

    /**
     * Получить журнал для новостей
     * @returns {Promise<Object?>}
     */
    get forNews() {
        const forNews = _.find(this.journals, journal => {
            return (journal.actual && journal.newsImportant);
        });

        return Promise.resolve(forNews);
    }

    /**
     * Получить все журналы
     * @returns {Promise<Object?>}
     */
    fetch() {
        if (!this.bunkerNode) {
            return Promise.resolve(null);
        }

        const { startIdx = 0 } = this._attributes;
        const lastIdx = startIdx + this._chunkLength;

        const filteredJournals = this._filterCollection(this.journals);
        const journalsWithDate = filteredJournals.filter('date');
        const journalsWithoutDate = filteredJournals.filter(journal => !journal.date).value();

        const list = journalsWithDate
            .sortBy(dateMonthYearSorter.bind(this._req, 'date', 'asc'))
            .reverse()
            .concat(journalsWithoutDate)
            .slice(startIdx, lastIdx)
            .value();

        const filtersData = _.get(this.bunkerNode, 'filters');

        return Promise.resolve({
            filters: this._getFiltersData(filtersData, this.journals),
            metaData: _.pick(this.bunkerNode, ['title', 'seo', 'og']),
            levelsData: _.get(this.bunkerNode, '_main'),
            regions: _.get(this.bunkerNode, 'regions'),
            more: filteredJournals.value().length > lastIdx,
            lastIdx,
            list
        });
    }

    /**
     * Добавляет данные для фильтрации в требуемом формате
     * @param {Object} entity
     * @returns {Object}
     */
    _assignFilters(entity) {
        const filters = filterSlugs
            .map(slug => {
                const value = _.get(entity, slug);
                const values = value ? [value] : [];

                return { slug, values };
            });

        return _.assign({ filters }, entity);
    }
}

module.exports = JournalsModel;
