'use strict';

const _ = require('lodash');

const NewsBase = require('./news-base');

/**
 * Модель для получения данных о блоге
 */
class Blog extends NewsBase {
    get MONTH_DICTIONARY() {
        return ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
    }

    /**
     * Преобразование данных модели после получения с сервера
     * @returns {Blog}
     * @private
     */
    _parseData() {
        const data = _(this._data)
            .pick(['slug',
                'postsCount',
                'localeTimezone',
                'socialLinks',
                'canComment',
                'hasRelatedArticles'
            ])
            .value();

        return this._addArchiveData(data);
    }

    /**
     * Получить информацию о блоге
     *
     * @returns {Blog}
     */
    fetch() {
        const options = {
            url: `/${this._attributes.blogIdentity}`,
            query: this._attributes.query
        };

        return this._request(options)
            .catch(error => {
                this._req.logger.error(error);

                return Promise.resolve({});
            });
    }

    /**
     * @param {Object} data блог
     * @returns {Blog}
     * @private
     */
    _addArchiveData(data) {
        const currentDate = new Date();
        const currentYear = currentDate.getFullYear();
        const currentMonth = currentDate.getMonth() + 1;

        const { query } = this._attributes;

        data.calendar = {
            MONTH_DICTIONARY: this.MONTH_DICTIONARY,
            start: this._getBlogStartDate(data.postsCount),
            current: {
                year: currentYear,
                month: currentMonth
            },
            checked: {
                year: query && query.year ? Number(query.year) : currentYear,
                month: query && query.month ? this.MONTH_DICTIONARY.indexOf(query.month) + 1 : null
            }
        };

        return data;
    }

    /* eslint-disable complexity */

    /**
     * Возвращает год и месяц первой записи в блоге
     * на основании поля postsCount.
     * @param {Object} postsCount
     * @returns {{year: Number, month: Number}}
     * @private
     */
    _getBlogStartDate(postsCount) {
        /* eslint-disable guard-for-in */
        let startYear = Infinity;
        let startMonth = Infinity;

        for (let year in postsCount) {
            year = Number(year);
            if (year < startYear) {
                startYear = year;
            }
        }

        for (const month in postsCount[startYear]) {
            const monthIdx = this.MONTH_DICTIONARY.indexOf(month);

            if (monthIdx >= 0 &&
                monthIdx < startMonth &&
                postsCount[startYear][month] > 0) {
                startMonth = monthIdx;
            }
        }

        return {
            year: startYear,
            month: startMonth + 1
        };
    }
}

module.exports = Blog;
