'use strict';

const config = require('yandex-cfg');
const _ = require('lodash');
const got = require('got');
const querystring = require('querystring');

// При слишком длинных запросах API Блогов отвечает с кодом 400
const { maxQueryLength } = config.blogs;

class Comments {
    constructor(req, attributes) {
        this._req = req;
        this._attributes = attributes;
    }

    init() {
        const options = {
            method: 'PUT',
            body: this._attributes.initData,
            headers: {
                'X-Ya-Service-Ticket': _.get(this._req, 'tvm.tickets.cmnt.ticket')
            }
        };

        const url = `${config.cmnt.api}${config.cmnt.initPath}`;

        return Comments
            ._request(url, options)
            .then(() => true)
            .catch(error => {
                this._req.logger.error(error, 'Failed to init cmnt chat');

                return false;
            });
    }

    sign() {
        const options = {
            method: 'PUT',
            body: this._attributes.signData,
            headers: {
                'X-Ya-Service-Ticket': _.get(this._req, 'tvm.tickets.cmnt.ticket')
            }
        };

        const url = `${config.cmnt.api}${config.cmnt.signPath}`;

        return Comments
            ._request(url, options)
            .then(({ body }) => _.get(body, 'signs.0.token', ''))
            .catch(error => {
                this._req.logger.error(error, 'Failed to sign data');

                return '';
            });
    }

    getCommentsCountByCmntIds(cmntIds) {
        if (cmntIds.length === 0) {
            return Promise.resolve({});
        }

        const url = `${config.cmnt.api}${config.cmnt.briefPath}`;

        const requests = _
            .chunk(cmntIds, config.cmnt.maxPostsSlugs)
            .map(chunk => {
                const options = {
                    headers: { 'X-Cmnt-Api-Key': config.cmnt.apiKey },
                    query: { entityId: chunk }
                };

                return Comments._request(url, options);
            });

        return Promise
            .all(requests)
            .then(responses => {
                return responses
                    .map(({ body }) => body.feed)
                    .reduce((acc, item) => {
                        Object.assign(acc, item);

                        return acc;
                    }, {});
            })
            .then(feed => {
                return _.mapValues(feed, commentsData => commentsData.count);
            })
            .catch(err => {
                this._req.logger.error(err);

                return {};
            });
    }

    getCommentsDataByBlogsSlugs() {
        let slugsToIds = null;

        return this
            ._getBlogsSlugsToCmntIds()
            .then(data => {
                slugsToIds = data;

                const cmntIds = _.values(slugsToIds);

                return this.getCommentsCountByCmntIds(cmntIds);
            })
            .then(idsToCommentsCount => {
                return _.mapValues(slugsToIds, postId => ({
                    count: idsToCommentsCount[postId]
                }));
            });
    }

    _getBlogsSlugsToCmntIds() {
        if (!this._attributes.slugs.length) {
            return Promise.resolve({});
        }

        const blogId = config.blogs.news[this._req.tld] || config.blogs.news.default;
        const baseQuery = {
            fields: '_id',
            lang: config.tldToBlogLang[this._req.tld]
        };

        const queryChunks = Comments._splitIntoQueryChunks(
            baseQuery,
            this._attributes.slugs,
            'slugs'
        );
        const options = {
            headers: {
                'X-Ya-Service-Ticket': _.get(this._req, 'tvm.tickets.blogs.ticket')
            }
        };

        const requests = queryChunks.map(query => {
            const url = `${config.blogs.host}/posts/custom/${blogId}?${query}`;

            return Comments._request(url, options);
        });

        return Promise
            .all(requests)
            .then(responses => {
                return responses
                    .map(({ body }) => _.mapValues(body, '_id'))
                    .reduce((acc, item) => {
                        Object.assign(acc, item);

                        return acc;
                    }, {});
            })
            .catch(error => {
                this._req.logger.error(error, 'Failed to load blog posts ids');

                return {};
            });
    }

    static _splitIntoQueryChunks(baseQuery, array, paramName) {
        const baseQueryStr = querystring.stringify({ ...baseQuery });
        const chunks = [];
        const queryParam = `&${paramName}=`;

        let queryChunk = baseQueryStr;

        for (const item of array) {
            if (queryChunk.length + queryParam.length + item.length > maxQueryLength) {
                chunks.push(queryChunk);

                queryChunk = baseQueryStr;
            }

            queryChunk += queryParam + item;
        }

        if (queryChunk) {
            chunks.push(queryChunk);
        }

        return chunks;
    }

    static _request(url, options) {
        const requestOptions = Object.assign({ headers: {} }, options);

        requestOptions.headers['Content-Type'] = 'application/json';
        requestOptions.json = true;

        return got(url, requestOptions);
    }
}

module.exports = Comments;
