'use strict';

const _ = require('lodash');
const config = require('yandex-cfg');
const got = require('got');

const Base = require('../abstractions/base');

/**
 *  Базовая модель для обращения к API блогов
 */
class NewsBase extends Base {

    /**
     * @param {Object} req параметры запроса
     * @param {Object} attributes атрибуты модели
     * @param {Object} [data] данные модели
     * @constructor
     */
    constructor(req, attributes, data) {
        super(req, attributes);

        this._data = data;
        this._headers = _.pick(this._req.headers, ['cookie', 'accept-encoding', 'content-type']);
    }

    getHeaders() {
        return this._headers;
    }

    /**
     * Экранирование опасных тегов для корректного вывода
     * @param {String} html
     * @returns {String}
     */
    sanitize(html) {
        return html
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;');
    }

    _removeEmptyProperties(obj) {
        for (const key in obj) {
            if (typeof obj[key] === 'undefined') {
                delete obj[key];
            }
        }

        return obj;
    }

    /**
     * Запрос на бекенд
     * @param {Object} options параметры запроса к бекенду,
     * @see: https://www.npmjs.com/package/request#request-options-callback
     * @private
     * @returns {Promise<NewsBase>}
     */
    _request(options) {
        options = options || {};
        options.method = options.method || 'GET';
        options.json = true;

        const headers = _.assign({}, options.headers);

        options.headers = this._removeEmptyProperties(headers);

        options.query = this._removeEmptyProperties(options.query);

        const url = config.blogs.host + options.url;

        delete options.url;

        options.headers['x-ya-service-ticket'] = _.get(this._req, 'tvm.tickets.blogs.ticket');

        return got(url, options)
            .then(this._onResponse.bind(this))
            .then(this._parseData.bind(this));
    }

    /**
     * Преобразование данных модели
     *
     * @returns {NewsBase}
     * @private
     */
    _parseData() {
        return this._data;
    }

    /**
     * Обработка ответа от бекенда
     *
     * @param {Object} data ответ от бекенда
     * @returns {NewsBase}
     * @protected
     */
    _onResponse(data) {
        this._data = data.body;

        return this;
    }
}

module.exports = NewsBase;
