'use strict';

const url = require('url');
const path = require('path');

const _ = require('lodash');

const NewsBase = require('./news-base');

/**
 * Модель для получения списка постов
 */
class Posts extends NewsBase {

    /**
     * Преобразование данных модели после получения с сервера
     * @returns {{ items: Posts, isPumpkinResponse: boolean }}
     * @private
     */
    _parseData() {
        const data = this._data.map(item => {
            return _(item)
                .pick([
                    '_id',
                    'slug',
                    'publishDate',
                    'approvedTitle',
                    'titleImage',
                    'viewType',
                    'approvedPreview',
                    'tags',
                    'hasNext',
                    'categoryIds'
                ])
                .value();
        });

        for (const post of data) {
            if (_.some(post.tags, { slug: 'hidden-hot-mark' })) {
                post.hasHotMark = true;
            }
        }

        const lastPost = data[data.length - 1];

        if (lastPost && lastPost.hasNext) {
            lastPost.nextUrl = url.format({
                pathname: this._buildAjaxPath('news-more', this._attributes.type),
                query: {
                    year: this._attributes.query.year,
                    month: this._attributes.query.month,
                    tag: this._attributes.query.tag,
                    categoryId: this._attributes.category,
                    from: lastPost._id
                }
            });
        }

        return { items: data, isPumpkinResponse: this._isPumpkinResponse };
    }

    /**
     * Получить список постов
     *
     * @returns {Posts}
     */
    /* eslint-disable complexity */
    fetch() {
        const commonQuery = ['from', 'size', 'lang', 'categoryId', 'tags'];

        let { query } = this._attributes;
        const requestUrl = {};

        const tag = query && query.tag;

        if (tag && typeof tag === 'string' && tag.startsWith('hidden-')) {
            return Promise.resolve([]);
        }

        if (query.year && query.month) {
            requestUrl.url = path.join('/posts/archive', this._attributes.blogIdentity);
            query = _.pick(query, _.union(['year', 'month'], commonQuery));
        } else {
            requestUrl.url = path.join('/posts', this._attributes.blogIdentity);
            query = _.pick(query, _.union(['tag'], commonQuery));
        }

        if (!_.isEmpty(query)) {
            requestUrl.query = query;
        }

        return this
            ._request(requestUrl)
            .catch(error => {
                this._req.logger.error(error);

                return [];
            });
    }

    _onResponse(data) {
        super._onResponse(data);

        this._isPumpkinResponse = Boolean(data.headers['x-is-pumpkin-response']);

        return this;
    }

    static unwrap(obj) {
        if (_.isPlainObject(obj.posts)) {
            const { items, isPumpkinResponse } = obj.posts;

            obj.posts = items;
            obj.isPumpkinResponse = isPumpkinResponse;
        }

        return obj;
    }
}

module.exports = Posts;
