'use strict';

const Promise = require('bluebird');
const _ = require('lodash');
const qs = require('qs');
const url = require('url');
const Base = require('./abstractions/base');
const helper = require('../lib/helper');

class Order extends Base {
    fetch() {
        const pages = helper.getBunkerNode(this._req.tld, this._bunker.sources, {
            path: 'order-pages'
        });
        const { pageName } = this._attributes;

        if (!Object.prototype.hasOwnProperty.call(pages, pageName)) {
            return Promise.reject({
                internalCode: '404_PNF',
                message: 'Order page not found'
            });
        }

        return Promise
            .resolve(pages[pageName])
            .bind(this)
            .then(this._parseData)
            .then(this._addAgenciesPath)
            .then(this._addSectionsUrls);
    }

    /**
     * Обрабатывает данные из бункера
     * @param {Object} page
     * @returns {Object}
     * @private
     */
    _parseData(page) {
        if (!page.enabled) {
            return this._checkRedirect(page);
        }

        const sections = _(page)
            .omit(['title', 'description', 'og', 'enabled', 'redirect'])
            .values()
            .value();

        const currentSection = this._attributes.sectionName || sections[0].id;
        const section = _.find(sections, { id: currentSection });

        if (!section) {
            return Promise.reject({
                internalCode: '404_SNF',
                message: 'Section not found'
            });
        }

        sections.forEach(entity => {
            entity.page = this._attributes.pageName;
        });

        return {
            currentSection,
            sections,
            page: _.pick(page, ['title', 'description', 'og'])
        };
    }

    _addAgenciesPath(data) {
        const agencySection = _.find(data.sections, { mod: 'agency' });

        if (agencySection) {
            agencySection.agenciesPath = this._buildPath('contact', 'agencies');
        }

        return data;
    }

    /**
     * Добавляет к каждой секции ее УРЛ
     * @param {Object} data
     * @returns {Array}
     * @private
     */
    _addSectionsUrls(data) {
        data.sections.forEach(section => {
            section.url = url.format({
                pathname: this._buildPath('order', this._attributes.pageName, section.id),
                search: decodeURIComponent(qs.stringify(this._req.query, {
                    arrayFormat: 'brackets'
                }))
            });
        });

        return data;
    }
}

module.exports = Order;
