'use strict';

const _ = require('lodash');
const config = require('yandex-cfg');

const helper = require('../lib/helper');
const { buildPrice } = require('../lib/prices');
const { isBrowserValid, removeEmojis } = require('../lib/helper');

const Base = require('./abstractions/base');

class Page extends Base {
    constructor(req, attributes) {
        super(req, attributes);

        this._pagesRelationsSection = _.get(this._req, ['pagesRelations', this.section]);
    }

    get section() {
        return _.get(this._attributes, 'section');
    }

    get page() {
        // Для всех страниц, которые лежат в папке landing (section нет в pagesRelations)
        // заполняем page из section, чтобы корректно работала продуктовая нода
        if (!this._pagesRelationsSection) {
            return this.section;
        }

        return _.get(this._attributes, 'page');
    }

    get notFoundError() {
        return {
            internalCode: '404_PNF',
            message: `Page /${this.section}/${this.page} was not found`
        };
    }

    get bunkerRoot() {
        return helper.getBunkerNode(this._req.tld, this._bunker.sources);
    }

    get sectionInfo() {
        return this._pagesRelationsSection || { node: 'landing' };
    }

    get sectionNode() {
        return _.get(this.bunkerRoot, this.sectionInfo.node);
    }

    get sectionsHandlers() {
        return {
            'product-price': this._handlePriceBlock.bind(this),
            'product-targetings': this._handleTargetingsBlock.bind(this),
            'product-similar': this._handleSimilarBlock.bind(this),
            'product-chart-line': this._handleChartLineBlock.bind(this)
        };
    }

    get menuRoot() {
        return this._buildPath(this.section);
    }

    getData() {
        // Для всех страниц, которые лежат в папке landing (section нет в pagesRelations)
        // Если заполнены и :section и :page (напр. /adv/turbo/turbo), возвращаем 404
        if (!this._pagesRelationsSection && _.get(this._req, 'params.page')) {
            return this._reject(this.notFoundError);
        }

        if (!this.sectionInfo.isProduct) {
            return this._getPage();
        }

        return this._reject(this.notFoundError);
    }

    getRss(feedData) {
        if (!this.sectionInfo.isProduct) {
            return this._getPageRss(feedData);
        }

        return this._reject(this.notFoundError);
    }

    _getPage() {
        const page = _.get(this.sectionNode, this.page, {});

        if (page.enabled) {
            return Promise.resolve(page)
                .then(this._getParentInfo.bind(this))
                .then(this._addMediaBlock.bind(this))
                .then(this._addProductInfo.bind(this))
                .then(this._sanitize.bind(this))
                .then(this._addQueryParams.bind(this))
                .then(this._addSections.bind(this))
                .catch(this._reject.bind(this, this.notFoundError));
        }

        if (page.redirect) {
            return this._reject({
                internalCode: '301_PMP',
                message: `Redirect from /${this.section}/${this.page}`,
                location: page.redirect
            });
        }

        return this._reject(this.notFoundError);
    }

    _getPageRss(feedData) {
        const page = _.get(this.sectionNode, this.page, {});
        const rssData = _.get(page, '_turbo', {});

        if (page.enabled && rssData.enabled) {
            feedData.item({
                _attr: { turbo: true },
                title: rssData.title,
                description: rssData.description,
                url: `${feedData.site_url}/${this.page}`,
                date: helper.getDate(rssData.pubDate),
                'custom_elements': [
                    { _attr: { turbo: true } },
                    { 'turbo:content': { _cdata: rssData.content.join(' ') } }
                ]
            });
        }
    }

    _getParentInfo(page) {
        if (page.parent) {
            return page;
        }

        const mainPage = _.get(this.sectionNode, '_main', {});

        return _.assign(page, {
            cta: _.get(page, 'cta.url') || Boolean(page.isAdProduct) ? page.cta : mainPage.cta,
            phone: _.get(page, 'phone.number') ? page.phone : mainPage.phone,
            sectionTitle: mainPage.sectionTitle
        });
    }

    _addMediaBlock(data) {
        data.media = _(['cases', 'stories', 'courses'])
            .reduce((result, value) => {
                result[value] = _.chain(this.bunkerRoot[`media-${value}`])
                    .filter(item => data.tags ? _.intersection(item.tags, data.tags).length : true)
                    .orderBy('date', 'desc')
                    .take(3)
                    .value();

                return result;
            }, {});

        return data;
    }

    _addProductInfo(data) {
        const isAdProduct = Boolean(data.isAdProduct);
        const fields = isAdProduct ?
            [
                'caption',
                'title',
                'text',
                'items',
                'cta',
                'requirements',
                'footer',
                'goals',
                'modal',
                'seo',
                'teaser'
            ] :
            [
                'title',
                'sectionTitle',
                'image',
                'imageWidth',
                'imageAlign',
                'text',
                'teaser',
                'cta',
                'phone',
                'noindex',
                'seo'
            ];

        data.product = _.pick(data, fields);

        data.regions = _.get(data, 'regions', {});

        return data;
    }

    _addSections(data) {
        data.sections = _(data)
            .values()
            .remove(section => {
                // Проверка, что поле продукта является объектом, имеет шаблон и флаг `enabled`
                return _.isObject(section) && _.has(section, 'block') && section.enabled;
            })
            .sortBy('order')
            .map(section => {
                const { block } = section;
                let sectionData = _.omit(section, ['block']);

                if (this.sectionsHandlers[block]) {
                    sectionData = this.sectionsHandlers[block](sectionData, data);
                }

                return {
                    block,
                    mix: { block: 'products-info', elem: 'section' },
                    data: sectionData,
                    productUrl: this._buildPath('products', this.section, this.page),
                    productsListUrl: this._buildPath('products')
                };
            })
            .value();

        return data;
    }

    _sanitize(data) {
        const needRemoveEmojis = !isBrowserValid(this._req.uatraits, config.emojisSupportBrowsers);

        if (Boolean(data.isAdProduct) && needRemoveEmojis) {
            const tabs = _(data)
                .get('product.modal.tabs', [])
                .map(tab => {
                    if (!tab.chat || !tab.chat.replicas) {
                        return tab;
                    }

                    const replicas = _(tab)
                        .get('chat.replicas', [])
                        .map(({ text, position }) => ({
                            text: removeEmojis(text),
                            position
                        }));

                    return { ...tab, chat: { ...tab.chat, replicas } };
                });

            data.product = { ...data.product, modal: { ...data.product.modal, tabs } };
        }

        return data;
    }

    _addQueryParams(data) {
        const isAdProduct = Boolean(data.isAdProduct);

        if (isAdProduct && this._req.query) {
            const params = {};

            if (typeof this._req.query.modal === 'string') {
                params.modal = this._req.query.modal;
            }

            if (typeof this._req.query.tab === 'string') {
                params.tab = this._req.query.tab;
            }

            data.queryParams = params;
        }

        return data;
    }

    _handlePriceBlock(sectionData, data) {
        const tablesPages = _(this.bunkerRoot['tables-pages'])
            .values()
            .flatten()
            .value() || [];

        const tables = (data.tables || []).concat(tablesPages);

        if (sectionData.priceId) {
            const price = this._getPriceById(sectionData.priceId);

            sectionData = _.merge({}, price, sectionData);
        }

        return buildPrice(tables, sectionData, _.get(this._req, 'locals.url'));
    }

    _getPriceById(priceId) {
        return _(this.bunkerRoot['product-prices'])
            .values()
            .find(price => price.priceId === priceId) || {};
    }

    _handleTargetingsBlock(sectionData) {
        const { targetings } = this.bunkerRoot;

        const targetingsList = sectionData.targetings.reduce((result, targeting) => {
            if (targetings[targeting]) {
                result.push(_.extend({ id: targeting }, targetings[targeting]));
            }

            return result;
        }, []);

        return _.set(sectionData, 'targetings', targetingsList);
    }

    _handleSimilarBlock(sectionData) {
        let products = {};

        // Пробегаем по всем продуктовым нодам и составляем общий список всех продуктов
        _.each(this._req.pagesRelations, node => {
            if (node.isProduct) {
                const sectionProducts = _.mapValues(this.bunkerRoot[node.node], product => {
                    // добавляем в каждый продукт поле 'section-slug', чтобы корректно
                    // сформировать ссылку на продукт из другого раздела
                    return _.set(product, 'section-slug', node.slug);
                });

                products = _.assign(products, sectionProducts);
            }
        });

        // В итоге products - объект, где ключ - уникальный id продукта, а значение - сам продукт
        const productsList = _.chain(sectionData.products)
            .reduce((result, slug) => {
                const productObj = products[slug];

                if (productObj && this.page !== slug) {
                    result.push({
                        title: productObj.title,
                        teaser: productObj.teaser,
                        section: productObj['section-slug'],
                        slug
                    });
                }

                return result;
            }, [])
            .shuffle()
            .take(3)
            .value();

        return _.set(sectionData, 'products', productsList);
    }

    _handleChartLineBlock(info) {
        const linesData = info.chart.data;
        const dates = linesData.map(lineData => {
            return lineData.values.map(point => point.date);
        });
        const datesIntersection = _.intersection(...dates);

        const result = linesData.map(lineData => {
            const modifiedLineData = {
                secondaryAxis: lineData.secondaryAxis ? 'secondaryAxis' : 'mainAxis',
                values: lineData.values.filter(point => _.includes(datesIntersection, point.date))
            };

            return _.assign({}, lineData, modifiedLineData);
        });

        const groupedLinesData = _(result)
            .groupBy('secondaryAxis')
            .defaults({ mainAxis: [], secondaryAxis: [] })
            .value();

        return _.merge({}, info, _.set({}, 'chart.data', groupedLinesData));
    }
}

module.exports = Page;
