'use strict';

const _ = require('lodash');
const Promise = require('bluebird');
const config = require('yandex-cfg');

const Base = require('./abstractions/base');

const CardTypesGenerator = require('../lib/card-types-generator');
const helper = require('../lib/helper');

const CHUNK_LENGTH = 8;
const CARD_TYPE_MODES = [{ count: 4, type: 'wide' }, { count: 3 }];

class Solutions extends Base {
    get isListPage() {
        return !this._attributes.slug;
    }

    get _type() {
        return this._attributes.type || 'solutions';
    }

    get _node() {
        return this._type;
    }

    get _baseUrlPath() {
        return this._type;
    }

    get _reactSections() {
        return {
            solutions: ['stories2', 'cases2', 'analytics2', 'practicums2'],
            'analytics-now': ['_self2']
        }[this._type];
    }

    get _sections() {
        const sections = {
            solutions: ['stories', 'cases', 'analytics', 'others', 'practicums'],
            'analytics-now': ['_self']
        }[this._type];

        return sections.concat(this._reactSections);
    }

    get _tld() {
        return this._attributes.tld || this._req.tld;
    }

    fetch() {
        const data = this._parseData({
            page: this.isListPage ? this._fetchPage() : null,
            section: this.isListPage ? null : this._fetchPage(),
            list: this._fetchSolutions()
        });

        return Promise.resolve(data).bind(this);
    }

    fetchCases(casesNodeName) {
        const { categoryId, lastCaseSlug, theme, type } = this._attributes.query;
        const { id, parentCategoryId, displayName } = config.caseCategory;

        if (theme || type
            || (categoryId !== id && categoryId !== parentCategoryId)
            || lastCaseSlug === null) {
            return [];
        }

        const solutionsNode = helper.getBunkerNode(this._tld,
            this._bunker.sources,
            { path: [this._node] }
        );

        const casesSettingsNodeName = casesNodeName.endsWith('2') ?
            casesNodeName.slice(0, -1) :
            casesNodeName;
        const { isDatesHidden } = solutionsNode[casesSettingsNodeName];

        const casesNode = solutionsNode[casesNodeName];

        const sortedCasesNodes = Object.keys(casesNode)
            .filter(node => casesNode[node].enabled)
            .sort((node1, node2) => {
                const firstDate = helper.getDate(casesNode[node1].date);
                const secondDate = helper.getDate(casesNode[node2].date);

                return secondDate - firstDate;
            });

        const lastIndex = _.findIndex(sortedCasesNodes,
            node => casesNode[node].slug === lastCaseSlug);
        const nextIndex = lastCaseSlug ? lastIndex + 1 : 0;
        const category = {
            url: `?categoryId=${id}`,
            parentCategoryId,
            displayName
        };

        return sortedCasesNodes
            .slice(nextIndex, nextIndex + config.limitMaterials)
            .map(this._caseToPost.bind(this, category, { casesNode, isDatesHidden }));
    }

    _caseToPost(category, casesData, nodeName) {
        const { casesNode, isDatesHidden } = casesData;
        const { title, date, image, slug } = casesNode[nodeName];

        return {
            slug,
            approvedTitle: title,
            titleImage: {
                w444: { fullPath: image }
            },
            date: helper.getDate(date),
            isDatesHidden,
            url: this._buildPath(this._baseUrlPath, 'cases', slug),
            categories: [category]
        };
    }

    _fetchPage() {
        const { section } = this._attributes;

        const solutionsPage = helper.getBunkerNode(
            this._tld,
            this._bunker.sources,
            { path: [this._node, section].filter(Boolean) }
        );

        return solutionsPage || {};
    }

    _fetchSolutions() {
        const { section } = this._attributes;

        const solutionsNode = helper.getBunkerNode(
            this._tld,
            this._bunker.sources,
            { path: [this._node] }
        );

        const reactSection = `${section}2`;
        const sectionProps = [section, reactSection];

        const sections = this.isListPage && section ? sectionProps : this._sections;

        const reactSlugs = new Set();

        solutionsNode[reactSection] = _.mapValues(
            solutionsNode[reactSection],
            solution => {
                if (solution.enabled) {
                    reactSlugs.add(solution.slug);
                }

                return Object.assign({ react: true }, solution);
            }
        );

        return _(solutionsNode)
            .pick(sections)
            .values()
            .map(eachSection => Object.values(eachSection))
            .flatten()
            .filter(solution => {
                const isEnabledObject = solution && solution.enabled;
                const hasReactCopy = !solution.react && reactSlugs.has(solution.slug);

                return (isEnabledObject || solution.redirect) && !hasReactCopy;
            })
            .value();
    }

    _parseData(data) {
        return this.isListPage
            ? this._parseSolutions(data)
            : this._parseSolution(data);
    }

    _parseSolutions(data) {
        const { section, filter } = this._attributes;

        const filters = _.castArray(filter || _.get(data, 'page.defaultFilter', {}));
        const isAnyCategory = filters.some(item => !item.value || item.value === 'all');

        const cardTypesGenerator = new CardTypesGenerator(CARD_TYPE_MODES);

        const allSolutions = [];

        for (const solution of _.orderBy(data.list, 'order', 'desc')) {
            const isRequiredSection = !section || section === solution.section;

            if (!isRequiredSection) {
                continue;
            }

            const isRequiredCategory = isAnyCategory || filters.every(({ by, value }) => {
                const solutionCategories = (solution[by] || '').split(',');

                return _.castArray(value).every(
                    filterValue => solutionCategories.includes(filterValue)
                );
            });

            if (!isRequiredCategory) {
                continue;
            }

            solution.cardType = cardTypesGenerator.next();

            allSolutions.push(solution);
        }

        const { startIdx = 0 } = this._attributes;
        const lastIdx = startIdx + CHUNK_LENGTH;

        const solutions = allSolutions.slice(startIdx, lastIdx);

        return {
            filter: filters[0],
            lastIdx,
            list: solutions,
            solutionsPage: data.page,
            more: lastIdx < allSolutions.length,
            hasMore: this._getHasMoreFilters(allSolutions)
        };
    }

    _getHasMoreFilters(allSolutions) {
        const hasMore = {};

        for (const category of Solutions.CATEGORIES) {
            hasMore[category] = _(allSolutions)
                .flatMap(solution => _.get(solution, category, '').split(','))
                .filter(Boolean)
                .uniq()
                .value();
        }

        return hasMore;
    }

    _parseSolution({ list, section }) {
        const { slug } = this._attributes;

        const solution =
            _.find(list, { slug, react: true })
            || _.find(list, { slug });

        return {
            solutionSection: section,
            solution
        };
    }
}

Solutions.CATEGORIES = ['tasks', 'industries', 'products', 'companySize', 'changes', 'periods'];

module.exports = Solutions;
