'use strict';

const express = require('express');
const config = require('yandex-cfg');
const routes = require('./routes');

/* eslint-disable new-cap */
const advRouter = express.Router();
const vdaRouter = express.Router();

function getRouters(route) {
    const routers = [advRouter];

    if (route.ajax) {
        routers.push(vdaRouter);
    }

    return routers;
}

/**
 * Подключает роутеры для /adv и /vda
 * @param {Object} app
 * @param {Function} configRouter
 */
function useRouters(app, configRouter) {
    app.use(config.router.adv, configRouter(advRouter));
    app.use(config.router.vda, configRouter(vdaRouter));
}

module.exports = function (app) {
    app.get('/adv/ping', (_, res) => res.send('pong'));

    routes.getRequest.forEach(route => {
        getRouters(route).forEach(router => {
            router.get(route.path, [], route.middleWares || [], route.controller);
        });
    });

    routes.postRequest.forEach(route => {
        getRouters(route).forEach(router => {
            router.post(route.path, route.middleWares || [], route.controller);
        });
    });

    routes.patchRequest.forEach(route => {
        getRouters(route).forEach(router => {
            router.patch(route.path, route.middleWares || [], route.controller);
        });
    });

    if (process.env.NODE_ENV === 'local') {
        const subdomain = require('express-subdomain');

        app.set('subdomain offset', 1);
        useRouters(app, router => subdomain(config.router.l7Subdomain, router));
        config.router.commonSubdomains.forEach(domain => app.use(subdomain(domain, advRouter)));
    } else {
        useRouters(app, router => router);
    }

    app.get('*', (req, res) => {
        res.sendStatus(404);
    });
};
