const captcha = require('../controllers/captcha');
const contacts = require('../controllers/contacts');
const edu = require('../controllers/edu');
const forms = require('../controllers/forms');
const freelancers = require('../controllers/freelancers');
const index = require('../controllers/index');
const instantArticles = require('../controllers/instant-articles');
const journals = require('../controllers/journals');
const mailRedirect = require('../controllers/mail-redirect');
const materials = require('../controllers/materials');
const news = require('../controllers/news.js');
const onlineCourses = require('../controllers/courses');
const onlineEdu = require('../controllers/online-edu');
const order = require('../controllers/order');
const page = require('../controllers/page');
const partners = require('../controllers/partners');
const prices = require('../controllers/prices');
const product = require('../controllers/product');
const promo = require('../controllers/promo');
const postcard = require('../controllers/postcard');
const publishers = require('../controllers/publishers');
const requirements = require('../controllers/requirements');
const search = require('../controllers/search');
const secretkeyCheck = require('../middleware/secretkey-check');
const solutions = require('../controllers/solutions');
const statvalue = require('../controllers/statvalue');
const subscription = require('../controllers/subscription');
const turbo = require('../controllers/turbo');
const { buildPath } = require('./path');

const checkCaptcha = require('../middleware/captcha');
const checkForm = require('../middleware/check-form');
const formData = require('../middleware/form-data');
const mainMenu = require('../middleware/main-menu');
const pagesRelations = require('../middleware/pages-relations');
const partnersAccess = require('../middleware/partners-access');
const sectionRegion = require('../middleware/section-region');

function getControllerInstance({ Controller, method }) {
    return function controller(req, res, next) {
        const instanceController = new Controller(req, res, next);

        instanceController[method]();
    };
}

const pageMiddleWares = [partnersAccess, mainMenu];

module.exports = {
    getRequest: [
        {
            path: '/',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: index,
                method: 'index'
            })
        },
        {
            path: '/statvalue',
            controller: getControllerInstance({
                Controller: statvalue,
                method: 'render'
            })
        },
        {
            path: '/promo/:slug',
            controller: getControllerInstance({
                Controller: promo,
                method: 'render'
            })
        },
        {
            path: '/promo/:slug/:pageId',
            controller: getControllerInstance({
                Controller: promo,
                method: 'render'
            })
        },
        {
            path: '/pozdravlenie',
            controller: getControllerInstance({
                Controller: postcard,
                method: 'render'
            })
        },
        {
            path: '/materials',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: materials,
                method: 'listPage'
            })
        },
        {
            path: '/journals',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: journals,
                method: 'list'
            })
        },
        {
            path: '/contact/',
            middleWares: [...pageMiddleWares, sectionRegion('contact')],
            controller: getControllerInstance({
                Controller: contacts,
                method: 'support'
            })
        },
        {
            path: '/contact/agencies',
            middleWares: [...pageMiddleWares, sectionRegion('contact')],
            controller: getControllerInstance({
                Controller: contacts,
                method: 'agencies'
            })
        },
        {
            path: '/contact/offices',
            middleWares: [...pageMiddleWares, sectionRegion('contact')],
            controller: getControllerInstance({
                Controller: contacts,
                method: 'offices'
            })
        },
        {
            path: '/contact/agencies/:slug',
            middleWares: [...pageMiddleWares, sectionRegion('contact')],
            controller: getControllerInstance({
                Controller: contacts,
                method: 'agency'
            })
        },
        {
            path: '/contact/agency',
            middleWares: [...pageMiddleWares, sectionRegion('contact')],
            controller: getControllerInstance({
                Controller: contacts,
                method: 'agenciesRedirect'
            })
        },
        {
            path: '/contact/agency/partner',
            middleWares: [...pageMiddleWares, sectionRegion('contact')],
            controller: getControllerInstance({
                Controller: contacts,
                method: 'agencyRedirect'
            })
        },
        {
            path: '/contact/freelancers',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: freelancers,
                method: 'list'
            })
        },
        {
            path: '/contact/freelancers/:slug',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: freelancers,
                method: 'freelancer'
            })
        },
        {
            path: '/order/:pageName',
            middleWares: [...pageMiddleWares],
            controller: order.index
        },
        {
            path: '/order/:pageName/:sectionName',
            middleWares: [...pageMiddleWares],
            controller: order.index
        },
        {
            path: '/edu',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: edu,
                method: 'index'
            })
        },
        {
            path: '/edu/events',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: edu,
                method: 'events'
            })
        },
        {
            path: '/edu/events/:event',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: edu,
                method: 'event'
            })
        },
        {
            path: '/edu/experts',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: edu,
                method: 'experts'
            })
        },
        {
            path: '/edu/experts/:expert',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: edu,
                method: 'expert'
            })
        },
        {
            path: '/edu/online',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: onlineEdu,
                method: 'index'
            })
        },
        {
            path: '/edu/online/:service',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: onlineEdu,
                method: 'service'
            })
        },
        {
            path: '/edu/online/:service/:lesson',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: onlineEdu,
                method: 'service'
            })
        },
        {
            path: '/edu/:service/:course',
            middleWares: [...pageMiddleWares, pagesRelations],
            controller: getControllerInstance({
                Controller: onlineCourses,
                method: 'course'
            })
        },
        {
            path: '/edu/:service/:course/:lesson',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: onlineCourses,
                method: 'course'
            })
        },
        {
            path: '/solutions',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: solutions,
                method: 'index'
            })
        },
        {
            path: '/solutions/stories',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: solutions,
                method: 'stories'
            })
        },
        {
            path: '/solutions/stories/:story',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: solutions,
                method: 'story'
            })
        },
        {
            path: '/solutions/cases',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: solutions,
                method: 'cases'
            })
        },
        {
            path: '/solutions/cases/:caseArticle',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: solutions,
                method: 'case'
            })
        },
        {
            path: '/solutions/practicums',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: solutions,
                method: 'practicums'
            })
        },
        {
            path: '/solutions/practicums/:practicum',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: solutions,
                method: 'practicum'
            })
        },
        {
            path: '/solutions/analytics',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: solutions,
                method: 'analytics'
            })
        },
        {
            path: '/solutions/analytics/:analyticsArticle',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: solutions,
                method: 'analyticsArticle'
            })
        },
        {
            path: '/solutions/others/:othersArticle',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: solutions,
                method: 'others'
            })
        },
        {
            path: '/analytics-now',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: solutions,
                method: 'analyticsNowIndex'
            })
        },
        {
            path: '/analytics-now/:selfArticle',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: solutions,
                method: 'analyticsNowArticle'
            })
        },
        {
            path: '/mail-redirect(/:postIdentity*)?',
            controller: getControllerInstance({
                Controller: mailRedirect,
                method: 'index'
            })
        },
        {
            path: '/products/:section',
            middleWares: [...pageMiddleWares, pagesRelations],
            controller: getControllerInstance({
                Controller: product,
                method: 'get'
            })
        },
        {
            path: '/products/:section/:page',
            middleWares: [...pageMiddleWares, pagesRelations],
            controller: getControllerInstance({
                Controller: product,
                method: 'get'
            })
        },
        {
            path: '/partners/',
            middleWares: [...pageMiddleWares, sectionRegion('partners')],
            controller: partners.main
        },
        {
            path: '/partners/enter/',
            middleWares: [...pageMiddleWares, sectionRegion('partners')],
            controller: partners.enter
        },
        {
            path: '/news-more/partners-news',
            ajax: true,
            middleWares: [partnersAccess],
            controller: partners.newsJson
        },
        {
            path: '/partners/news/',
            controller (req, res) {
                res.redirect(buildPath(req.locals.url, 'partners'));
            }
        },
        {
            path: '/partners/subscription',
            middleWares: [...pageMiddleWares, sectionRegion('partners')],
            controller: subscription.partners
        },
        {
            path: '/partners/:pageName',
            middleWares: [...pageMiddleWares, sectionRegion('partners')],
            controller: partners.partners
        },
        {
            path: '/partners/news/:post',
            middleWares: [...pageMiddleWares, sectionRegion('partners')],
            controller: partners.post
        },
        {
            path: '/partners/:blockName/:pageName',
            middleWares: [...pageMiddleWares, sectionRegion('partners')],
            controller: partners.partners
        },
        {
            path: '/news/',
            middleWares: [...pageMiddleWares, sectionRegion('news')],
            controller: news.listPage
        },
        {
            path: '/captcha/new',
            ajax: true,
            controller: getControllerInstance({
                Controller: captcha,
                method: 'showCaptcha'
            })
        },
        {
            path: '/news-more/news',
            ajax: true,
            controller: news.listJson
        },
        {
            path: '/news/:post',
            middleWares: [...pageMiddleWares, sectionRegion('news')],
            controller: news.post
        },
        {
            path: '/news/:blogIdentity/:postIdentity/relatedArticles',
            ajax: true,
            controller: news.relatedArticles
        },
        {
            path: '/prices-common/',
            middleWares: [...pageMiddleWares, pagesRelations, sectionRegion('prices-common')],
            controller: prices.common
        },
        {
            path: '/prices/',
            middleWares: [...pageMiddleWares, pagesRelations, sectionRegion('prices')],
            controller: prices.list
        },
        {
            path: '/publishers',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: publishers,
                method: 'index'
            })
        },
        {
            path: '/publishers/subscription',
            middleWares: [...pageMiddleWares, sectionRegion('publishers')],
            controller: subscription.publishers
        },
        {
            path: '/requirements/',
            middleWares: [...pageMiddleWares, sectionRegion('requirements')],
            controller: requirements.page
        },
        {
            path: '/requirements/:pageName',
            middleWares: [...pageMiddleWares, sectionRegion('requirements')],
            controller: requirements.page
        },
        {
            path: '/subs',
            middleWares: [...pageMiddleWares, sectionRegion('subscription')],
            controller: subscription.subs
        },
        {
            path: '/subscription',
            middleWares: [...pageMiddleWares, sectionRegion('subscription')],
            controller: subscription.public
        },
        {
            path: '/turboRss',
            middleWares: [pagesRelations],
            controller: getControllerInstance({
                Controller: turbo,
                method: 'rss'
            })
        },
        {
            path: '/instantArticlesRss',
            controller: getControllerInstance({
                Controller: instantArticles,
                method: 'rss'
            })
        },
        {
            path: '/search',
            middleWares: [...pageMiddleWares],
            controller: getControllerInstance({
                Controller: search,
                method: 'serp'
            })
        },
        {
            path: '/:section/:page',
            middleWares: [...pageMiddleWares, pagesRelations],
            controller: getControllerInstance({
                Controller: page,
                method: 'get'
            })
        },
        {
            path: '/:section',
            middleWares: [...pageMiddleWares, pagesRelations],
            controller: getControllerInstance({
                Controller: page,
                method: 'get'
            })
        }
    ],
    postRequest: [
        {
            path: '/send/',
            ajax: true,
            middleWares: [secretkeyCheck, formData, checkForm, checkCaptcha],
            controller: forms.contactForm
        },
        {
            path: '/subscribe/',
            ajax: true,
            middleWares: [secretkeyCheck, formData, checkForm],
            controller: subscription.request
        },
        {
            path: '/transaction/',
            ajax: true,
            middleWares: [secretkeyCheck, formData, checkForm],
            controller: subscription.transaction
        },
        {
            path: '/partners/send/',
            ajax: true,
            middleWares: [secretkeyCheck],
            controller: forms.partnersForm
        },
        {
            path: '/solutions-filter',
            ajax: true,
            controller: getControllerInstance({
                Controller: solutions,
                method: 'filter'
            })
        },
        {
            path: '/analytics-now-filter',
            ajax: true,
            controller: getControllerInstance({
                Controller: solutions,
                method: 'analyticsNowFilter'
            })
        },
        {
            path: '/journals-filter',
            ajax: true,
            controller: getControllerInstance({
                Controller: journals,
                method: 'filter'
            })
        },
        {
            path: '/contact-freelancer',
            ajax: true,
            middleWares: [formData, checkForm],
            controller: getControllerInstance({
                Controller: freelancers,
                method: 'contact'
            })
        },
        {
            path: '/materials-filter',
            ajax: true,
            controller: getControllerInstance({
                Controller: materials,
                method: 'filter'
            })
        },
        {
            path: '/statvalue',
            ajax: true,
            middleWares: [secretkeyCheck],
            controller: getControllerInstance({
                Controller: statvalue,
                method: 'calculate'
            })
        },
        {
            path: '/companies',
            ajax: true,
            middleWares: [secretkeyCheck],
            controller: getControllerInstance({
                Controller: contacts,
                method: 'filterAgencies'
            })
        },
        {
            path: '/company',
            ajax: true,
            middlewares: [secretkeyCheck],
            controller: getControllerInstance({
                Controller: contacts,
                method: 'agencyData'
            })
        }
    ],
    patchRequest: [
        {
            path: '/company',
            ajax: true,
            middlewares: [secretkeyCheck],
            controller: getControllerInstance({
                Controller: contacts,
                method: 'editAgency'
            })
        }
    ]
};
