'use strict';

let _ = require('lodash');
let LevelManager = require('./levels');
let GeminiPlugin = require('./gemini');
let makeConfig = require('../config');

const DEFAULT_PLATFORMS = {
    desktop: ['common', 'desktop'],
    'touch-phone': ['common', 'touch-phone']
};
const DEFAULT_LANGUAGES = ['ru', 'en', 'tr'];

module.exports = function (config) {
    function configurePlatform(config, platform, options) {
        let levels = new LevelManager(config, options.platforms, options.libraries);
        let plugin;

        if (options.gemini) {
            let geminiPath = platform + '.gemini';
            plugin = new GeminiPlugin(`gemini-${platform}`, config, {
                destPath: geminiPath,
                target: 'common'
            });

            let geminiConfigurator = options.gemini.configurator;

            plugin.addConfigurator(config => {
                geminiConfigurator.configure(config, {
                    platform,
                    destPath: geminiPath,
                    levels: levels.getLevels(platform, { tech: 'gemini' }),
                    sourceLevels: levels.getSourceLevels(platform)
                });
            });
        }

        if (options.bundles) {
            let bundlesConfigurator = options.bundles.configurator;
            let isBundlePlatform = _.includes(makeConfig.BUNDLES_PLATFORMS, platform);
            if (bundlesConfigurator && isBundlePlatform) {
                bundlesConfigurator.configure(config, {
                    platform,
                    destPath: platform + '.bundles',
                    levels: levels.getLevels(platform, { tech: 'bundles' }),
                    sourceLevels: levels.getSourceLevels(platform)
                });
            }
        }
    }

    function configureI18N(options) {
        config.includeConfig('enb-tanker/enb-make');
        config.setEnv(options.i18n);
    }

    config.registerModule('enb-tools', {
        configureProject(options) {
            _.defaults(options, {
                libraries: [],
                platforms: DEFAULT_PLATFORMS,
                languages: DEFAULT_LANGUAGES,
                gemini: false
            });

            config.setLanguages(options.languages);

            let platformNames = Object.keys(options.platforms);
            platformNames.forEach(platform => {
                configurePlatform(config, platform, options);
            });

            configureI18N(options);
        }
    });
};
