'use strict';

let fs = require('fs');
let path = require('path');

let _ = require('lodash');

function LevelManager(config, platforms, libs) {
    this._config = config;
    this._platforms = platforms;

    this._libs = libs || [];

    this._levels = {};

    this._buildLevels();
}

LevelManager.prototype._buildLevels = function () {
    let platforms = this._platforms;

    Object.keys(platforms).forEach(function (platform) {
        this._buildLevelsFor(platform);
    }, this);
};

LevelManager.prototype._buildLevelsFor = function (platform) {
    let self = this;
    let libs = this._libs;

    this._levels[platform] = (libs || [])
        .filter(lib => {
            return !lib.platform || lib.platform === platform;
        })
        .reduce((levels, lib) => {
            let libLevels = lib.levels ?
                _.clone(lib.levels) :
                self._buildLibLevels(lib, platform);

            return levels.concat(libLevels);
        }, []);
};

LevelManager.prototype._buildLibLevels = function (lib, platform) {
    let config = this._config;

    let levelPattern = lib.levelPattern || '*.blocks';
    let levelNames = this._platforms[platform];

    let names = (lib.platforms && lib.platforms[platform]) || levelNames;

    return names
        .map(name => {
            return levelPattern.replace('*', name);
        })
        .reduce((levels, level) => {
            let levelPath = lib.path || '';
            levelPath = config.resolvePath(path.join(levelPath, level));

            if (fs.existsSync(levelPath)) {
                let isSource = Boolean(lib.source);

                levels.push({
                    path: levelPath,
                    tech: lib.tech,
                    check: Boolean(lib.check) || isSource,
                    source: isSource
                });
            }

            return levels;
        }, []);
};

LevelManager.prototype.filterLevelsByTech = function (levels, tech) {
    return tech ?
        levels.filter(level => {
            return !level.tech || level.tech === tech;
        }) :
        levels;
};

LevelManager.prototype.getSourceLevels = function (platform) {
    let levels = this._levels[platform] || [];

    return levels.filter(level => {
        return level.source;
    });
};

LevelManager.prototype.getLevels = function (platform, options) {
    options = options || {};

    let levels = this._levels[platform] || [];

    return this.filterLevelsByTech(levels, options.tech);
};

module.exports = LevelManager;
