'use strict';

/**
 * gemini
 * ======
 *
 * Склеивает *gemini*-файлы по deps'ам в виде `?.gemini.js`.
 * Предназначен для сборки gemini тестов.
 *
 * **Опции**
 *
 * * *String* **target** — Результирующий таргет. По умолчанию — `?.gemini.js`.
 * * *String* **dirsTarget** — dirs-таргет, на основе которого получается список `?.priv-gemini.js` файлов (его предоставляет технология `files`). По умолчанию — `?.dirs`.
 * * *String* **blocksPath** —  Путь к диретории html файлов для тестов.
 * * *String* **techSuffix** — Cуфикс целевых прив-файлов. По умолчанию — `gemini.js`.
 * * *String* **blockPrefix** — Префикс для прив-файлов с тестами. По умолчанию — `gemini-`.
 *
 * **Пример**
 *
 * ```javascript
 * nodeConfig.addTech(require('enb/techs/priv-gemini'), {
 *     blocksPath: nodeConfig.resolvePath(path.join('desktop.gemini', 'html'))
 * });
 * ```
 */

const path = require('path');
const Vow = require('vow');
const vfs = require('enb/lib/fs/async-fs');

module.exports = require('enb/lib/build-flow').create()
    .name('gemini')
    .target('target', '?.gemini.js')
    .defineRequiredOption('blocksPath')
    .defineOption('techSuffix', 'gemini.js')
    .defineOption('blockPrefix', 'gemini-')
    .useDirList('gemini')
    .builder(function (dirs) {
        let node = this.node;
        let blocksPath = this._blocksPath;
        let techSuffix = this._techSuffix;
        let blockPrefix = this._blockPrefix;
        let rootUrl = path.parse(blocksPath).name;
        let suites = {};

        let promises = dirs.map(dir => {
            return getDirContent(node, dir, techSuffix).then(files => {
                files.forEach(file => {
                    if (!file) {
                        return;
                    }

                    let suite = path.basename(file.name, '.' + file.suffix);

                    if (!suites[suite]) {
                        suites[suite] = [];
                    }

                    suites[suite].push(file.content);
                });
            });
        });

        return Vow.all(promises).then(() => {
            return renderFile(suites, rootUrl, blockPrefix);
        });
    })
    .createTech();

/**
 * Считываем все тесты из директории
 * @param {Object} node
 * @param {String} dir
 * @param {string} techSuffix
 * @returns {Promise<Object[]>}
 */
function getDirContent(node, dir, techSuffix) {
    return Vow.all(dir.files.map(file => {
        if (file.suffix !== techSuffix) {
            return undefined;
        }

        return getFileContent(node, file.fullname).then(content => {
            return {
                name: file.fullname,
                suffix: file.suffix,
                content
            };
        });
    }));
}

/**
 * Получаем код одного теста
 * @param {Object} node
 * @param {String} file
 * @returns {Promise<String>}
 */
function getFileContent(node, file) {
    return vfs.read(file, 'utf8').then(data => {
        let relPath = node.relativePath(file);

        return `
            // begin: ${relPath}
            (function () {
                ${data}
            }());
            // end: ${relPath}
        `;
    });
}

/**
 * Генерируем gemini файл со всеми тестами
 * @param {Object} suites
 * @param {String} rootUrl
 * @param {String} blockPrefix
 * @returns {String}
 */
function renderFile(suites, rootUrl, blockPrefix) {
    return Object.keys(suites).map(suite => {
        let suiteData = suites[suite].join('\n');
        return renderSuite(suite, suiteData, rootUrl, blockPrefix);
    }).join('\n');
}

/**
 * Генерируем один gemini тест
 * @param {String} name Название теста
 * @param {String} data Код теста
 * @param {String} rootUrl
 * @param {String} blockPrefix
 * @returns {String}
 */
function renderSuite(name, data, rootUrl, blockPrefix) {
    let blockName = blockPrefix + name;
    let url = [rootUrl, name, name + '.html'].join('/');

    return `
        // begin suite: ${name},
        (function () {
            gemini.suite('${name}', function (suite) {
            var blockName = '${blockName}';
            suite = suite
                .setUrl('/${url}')
                .setCaptureElements('.${blockName}')
                .before(function (actions, find) {
                    actions.mouseMove(find('body'), { x:0, y:0 });
                });
              
                ${data}
            });
        }());
        // end suite: ${name}
    `;
}
