'use strict';

/**
 * priv-gemini
 * ===========
 *
 * Генерирует набор статичных html файлов на основе *priv-gemini*-файлы.
 * Предназначен для сборки статичных страниц для тестирования скриншотами.
 *
 * **Опции**
 *
 * * *String* **target** — Результирующий таргет. По умолчанию — `?.priv.gemimi.js`. Будет содержать список сгенерированных html файлов.
 * * *String* **privFile** — Абсолютный путь к собранному `.priv.js` файлу. По умолчанию — `?.priv.js`.
 * * *String* **bhFile** — Абсолютный путь к собранному `.bh.js` файлу. По умолчанию — `?.bh.js`.
 * * *String* **destPath** — Путь к диретории результирующих html файлов.
 * * *String* **dirsTarget** — dirs-таргет, на основе которого получается список `?.priv-gemini.js` файлов (его предоставляет технология `files`). По умолчанию — `?.dirs`.
 * * *String* **pageBlockName** — Название блока для оборачивания всех тестируемых привов.
 * * *String* **techSuffix** — Cуфикс целевых прив-файлов. По умолчанию — `priv-gemini.js`.
 * * *String* **blockPrefix** — Префикс для прив-файлов с тестами. По умолчанию — `gemini-`.
 *
 * **Пример**
 *
 * ```javascript
 * nodeConfig.addTech(require('enb/techs/priv-gemini'), {
 *     destPath: nodeConfig.resolvePath(path.join('desktop.gemini', 'html'))
 * });
 * ```
 */

const path = require('path');
const Vow = require('vow');
const vfs = require('enb/lib/fs/async-fs');

module.exports = require('enb/lib/build-flow').create()
    .name('priv-gemini')
    .target('target', '?.priv.gemini.js')
    .defineRequiredOption('privFile')
    .defineRequiredOption('bhFile')
    .defineRequiredOption('destPath')
    .defineOption('pageBlockName')
    .defineOption('techSuffix', 'priv-gemini.js')
    .defineOption('blockPrefix', 'gemini-')
    .useSourceFilename('privFile', '?.priv.js')
    .useSourceFilename('bhFile', '?.bh.js')
    .useDirList('gemini')
    .builder(function (privFile, bhFile, dirs) {
        let node = this.node;
        let blocks = require(privFile);
        let bh = require(bhFile);
        let destPath = node.resolvePath(this._destPath);
        let pageBlockName = this._pageBlockName;
        let techSuffix = this._techSuffix;
        let blockPrefix = this._blockPrefix;

        let blockNames = getBlockNames(blocks, dirs, techSuffix, blockPrefix);
        let promises = [];

        try {
            promises = blockNames.map(blockName => {
                let bemjson = renderBlock(blocks, blockName, pageBlockName);
                let html = bh.apply(bemjson);
                let testName = blockName.slice(blockPrefix.length);
                let dirName = path.join(destPath, testName);
                let fileName = path.join(dirName, testName + '.html');

                return vfs.makeDir(dirName).then(() => {
                    return saveHtml(node, fileName, html);
                });
            });
        } catch (err) {
            console.error(err);
            // todo: ENB 0.17 не ловит rejected промисы...
            process.exit(1);
            return Vow.reject(err);
        }

        return Vow.all(promises)
            .then(files => {
                return '/* ' + files.join('*/\n/*') + ' */';
            })
            .catch(err => {
                console.error(err);
                // todo: ENB 0.17 не ловит rejected промисы...
                process.exit(1);
                throw err;
            });
    })
    .createTech();

/**
 * Получаем из списка директорий список блоков для тестирования
 * @param {Blocks} blocks
 * @param {String[]} dirs
 * @param {String} techSuffix
 * @param {String} blockPrefix
 * @returns {String[]}
 */
function getBlockNames(blocks, dirs, techSuffix, blockPrefix) {
    return dirs.reduce((blockNames, dir) => {
        return dir.files.reduce((blockNames, file) => {
            if (file.suffix !== techSuffix) {
                return blockNames;
            }

            let testName = path.basename(file.name, '.' + file.suffix);
            let blockName = blockPrefix + testName;

            // register gemini priv
            require(file.fullname)(blocks, blockName);

            if (blockNames.indexOf(blockName) === -1) {
                blockNames.push(blockName);
            }

            return blockNames;
        }, blockNames);
    }, []);
}

/**
 * @param {Blocks} blocks
 * @param {String} blockName
 * @param {String} [pageBlockName]
 * @returns {Object}
 */
function renderBlock(blocks, blockName, pageBlockName) {
    if (pageBlockName) {
        return blocks.exec(pageBlockName, {
            title: 'Test page for ' + blockName,
            blockName
        });
    } else {
        return blocks.exec(blockName);
    }
}

/**
 * @param {Object} node
 * @param {String} fileName
 * @param {String} html
 * @returns {Promise<String>}
 */
function saveHtml(node, fileName, html) {
    vfs.write(fileName, html, 'utf-8').then(() => {
        return node.relativePath(fileName);
    });
}
