'use strict';

let techs = require('./techs');
let autoprefixer = require('./autoprefixer');

module.exports = {
    configure(config, options) {
        let platform = options.platform;
        let levels = options.levels;

        config.nodes(platform + '.bundles/*', nodeConfig => {

            nodeConfig.mode('development', () => {
                nodeConfig.addTechs([
                    [techs.borschik, {
                        sourceTarget: '?.{lang}.js',
                        destTarget: '_?.{lang}.js',
                        minify: false,
                        freeze: false
                    }],

                    [techs.fileCopy, { sourceTarget: '?.css', destTarget: '_?.css' }],
                    [techs.fileCopy, { sourceTarget: '?.ie9.css', destTarget: '_?.ie9.css' }]
                ]);
            });

            nodeConfig.mode('production', () => {
                nodeConfig.addTechs([
                    [techs.borschik, {
                        sourceTarget: '?.{lang}.js',
                        destTarget: '_?.{lang}.js',
                        minify: true,
                        freeze: true
                    }],

                    [techs.borschik, {
                        sourceTarget: '?.css',
                        destTarget: '_?.css',
                        minify: true,
                        freeze: true,
                        tech: 'cleancss',
                        techOptions: {
                            cleancss: { advanced: false }
                        }
                    }],
                    [techs.borschik, {
                        sourceTarget: '?.ie9.css',
                        destTarget: '_?.ie9.css',
                        minify: true,
                        freeze: true,
                        tech: 'cleancss',
                        techOptions: {
                            compatibility: 'ie9',
                            cleancss: { advanced: false }
                        }
                    }]
                ]);

                // копируем все необходимые собраные файлы в конечную папку
                nodeConfig.addTargets([
                    'public',
                    'i18n-static.{lang}',
                    'i18n-public.{lang}'
                ]);

                nodeConfig.addTechs([
                    [techs.smartCopy, {
                        sourceTargets: ['_?.css', '_?.ie9.css'],
                        dest: '../out/public',
                        target: 'public'
                    }],
                    [techs.smartCopy, {
                        sourceTargets: ['_?.{lang}.js'],
                        dest: '../out/public',
                        target: 'i18n-public.{lang}',
                        lang: '{lang}'
                    }],
                    [techs.smartCopy, {
                        sourceTargets: ['?.{lang}.priv.js', '?.{lang}.bh.js'],
                        dest: '../out/static',
                        target: 'i18n-static.{lang}',
                        lang: '{lang}'
                    }]
                ]);
            });

            nodeConfig.addTechs([
                // Сканируем левелы
                [techs.levels, { levels }],

                // Собираем bemdecl-файлы
                [techs.fileProvider, { target: '?.bemdecl.js' }],
                [techs.depsByTechToBemdecl, {
                    sourceSuffixes: 'deps.js',
                    target: '?.client.bemdecl.js',
                    sourceTech: 'js',
                    destTech: 'bemhtml'
                }],

                // Строим зависимости
                [techs.deps],
                [techs.deps, {
                    bemdeclFile: '?.client.bemdecl.js',
                    target: '?.client.deps.js'
                }],

                // Строим список файлов
                [techs.files],
                [techs.files, {
                    depsFile: '?.client.deps.js',
                    filesTarget: '?.client.files',
                    dirsTarget: '?.client.dirs'
                }],

                // Собираем файлы на основе *.i18n-папок для указанных языков.
                [techs.i18nKeysets, { lang: '{lang}' }],
                [techs.i18nLang, {
                    lang: '{lang}',
                    exports: {
                        globals: 'force', // force нужно форсить глобальный экспорт i18n для BH шаблонов
                        commonJS: false, // без этой опции технология i18n ломает export BH из .{lang}.bemhtml.js
                        ym: false // модульная система (пока) не используется
                    }
                }],

                // Собираем клиентские bh и priv файлы
                [techs.bhBundle, {
                    filesTarget: '?.client.files',
                    target: '?.client.bh.js',
                    mimic: ['BEMHTML'],
                    mode: 'production',
                    bhOptions: {
                        jsAttrName: 'data-bem',
                        jsAttrScheme: 'json',
                        jsElem: false
                    },
                    requires: {
                        i18n: {
                            globals: 'BEM.I18N'
                        }
                    }
                }],

                // Собираем клиентский js
                [techs.js],
                [techs.fileMerge, {
                    lang: '{lang}',
                    sources: ['?.browser.js', '?.lang.{lang}.js', '?.client.bh.js'],
                    target: '?.{lang}.js'
                }],

                // Собираем серверные bh
                [techs.bhBundle, {
                    target: '?.bh.js',
                    mimic: ['BEMHTML'],
                    mode: 'production',
                    bhOptions: {
                        jsAttrName: 'data-bem',
                        jsAttrScheme: 'json',
                        jsElem: false
                    },
                    requires: {
                        i18n: {
                            globals: 'BEM.I18N'
                        }
                    }
                }],
                [techs.fileMerge, {
                    lang: '{lang}',
                    sources: ['?.lang.{lang}.js', '?.bh.js'],
                    target: '?.{lang}.bh.js'
                }],

                // Собираем серверный priv
                [techs.privServer],
                [techs.fileMerge, {
                    lang: '{lang}',
                    sources: ['?.lang.{lang}.js', '?.priv.js'],
                    target: '?.{lang}.priv.js'
                }],

                // Собираем CSS
                [techs.css, {
                    autoprefixer: { browsers: autoprefixer.getBrowsers(platform, 'css') }
                }],
                [techs.css, {
                    target: '?.ie9.css',
                    sourceSuffixes: ['styl', 'css', 'ie.styl', 'ie.css', 'ie9.styl', 'ie9.css'],
                    autoprefixer: { browsers: autoprefixer.getBrowsers(platform, 'ie9.css') }
                }]
            ]);

            nodeConfig.addTargets([
                '_?.{lang}.js',
                '?.{lang}.priv.js',
                '?.{lang}.bh.js',

                '_?.css',
                '_?.ie9.css'
            ]);
        });
    }
};
