'use strict';

let path = require('path');
let techs = require('./techs');
let autoprefixer = require('./autoprefixer');

let gemini = require('./../enb/techs/gemini');
let privGemini = require('./../enb/techs/priv-gemini');

module.exports = {
    configure(config, options) {
        let platform = options.platform;
        let levels = options.levels;
        let sourceLevels = options.sourceLevels;

        config.nodes(platform + '.gemini/common', nodeConfig => {
            nodeConfig.addTechs([
                // Сканируем левелы
                [techs.levels, { levels }],
                [techs.levels, {
                    levels: sourceLevels,
                    target: '?.source.levels'
                }],

                // Собираем bemdecl-файлы
                [techs.fileProvider, { target: '?.bemdecl.js' }],
                [techs.depsByTechToBemdecl, {
                    target: '?.client.bemdecl.js',
                    sourceTech: 'js',
                    destTech: 'bemhtml'
                }],
                [techs.levelsToBemdecl, {
                    source: '?.source.levels',
                    target: '?.source.bemdecl.js'
                }],
                [techs.mergeBemdecl, {
                    target: '?.all.bemdecl.js',
                    sources: ['?.bemdecl.js', '?.source.bemdecl.js']
                }],

                // Строим зависимости
                [techs.deps, { bemdeclFile: '?.all.bemdecl.js' }],
                [techs.deps, {
                    bemdeclFile: '?.client.bemdecl.js',
                    target: '?.client.deps.js'
                }],
                [techs.deps, {
                    bemdeclFile: '?.source.bemdecl.js',
                    levelsTarget: '?.source.levels',
                    target: '?.source.deps.js'
                }],

                // Строим список файлов
                [techs.files],
                [techs.files, {
                    depsFile: '?.client.deps.js',
                    filesTarget: '?.client.files',
                    dirsTarget: '?.client.dirs'
                }],
                [techs.files, {
                    depsFile: '?.source.deps.js',
                    levelsTarget: '?.source.levels',
                    filesTarget: '?.source.files',
                    dirsTarget: '?.source.dirs'
                }],

                // Собираем файлы на основе *.i18n-папок для указанных языков.
                [techs.i18nKeysets, { lang: '{lang}' }],
                [techs.i18nLang, {
                    lang: '{lang}',
                    exports: {
                        globals: 'force', // force нужно форсить глобальный экспорт i18n для BH шаблонов
                        commonJS: false, // без этой опции технология i18n ломает export BH из .{lang}.bemhtml.js
                        ym: false // модульная система (пока) не используется
                    }
                }],

                // Собираем клиентские bh и priv файлы
                [techs.bhBundle, {
                    filesTarget: '?.client.files',
                    target: '?.client.bh.js',
                    mimic: ['BEMHTML'],
                    mode: 'production',
                    bhOptions: {
                        jsAttrName: 'data-bem',
                        jsAttrScheme: 'json',
                        jsElem: false
                    },
                    requires: {
                        i18n: {
                            globals: 'BEM.I18N'
                        }
                    }
                }],

                // Собираем клиентский js
                [techs.js, { target: '?.pre.js' }],
                [techs.fileMerge, {
                    sources: [
                        '?.pre.js',
                        '?.lang.ru.js',
                        '?.client.bh.js'
                    ],
                    target: '?.js'
                }],

                // Собираем серверные bh
                [techs.bhBundle, {
                    target: '?.bh.js',
                    mimic: ['BEMHTML'],
                    mode: 'production',
                    bhOptions: {
                        jsAttrName: 'data-bem',
                        jsAttrScheme: 'json',
                        jsElem: false
                    },
                    requires: {
                        i18n: {
                            globals: 'BEM.I18N'
                        }
                    }
                }],
                [techs.fileMerge, {
                    lang: '{lang}',
                    sources: ['?.lang.{lang}.js', '?.bh.js'],
                    target: '?.{lang}.bh.js'
                }],

                // Собираем серверный priv
                [techs.privServer],
                [techs.fileMerge, {
                    lang: '{lang}',
                    sources: ['?.lang.{lang}.js', '?.priv.js'],
                    target: '?.{lang}.priv.js'
                }],

                // Собираем gemini
                [privGemini, {
                    destPath: path.join('..', '..', 'gemini', `${platform}.gemini`, 'html'),
                    pageBlockName: 'gemini-page',
                    privFile: '?.ru.priv.js',
                    bhFile: '?.ru.bh.js',
                    dirsTarget: '?.source.dirs',
                    target: '?.priv.gemini.js'
                }],
                [gemini, {
                    blocksPath: config.resolvePath(path.join(`${platform}.gemini`, 'html')),
                    dirsTarget: '?.source.dirs',
                    target: '?.gemini.js'
                }],

                // Собираем CSS
                [techs.css, {
                    autoprefixer: { browsers: autoprefixer.getBrowsers(platform, 'css') }
                }],
                [techs.css, {
                    target: '?.ie9.css',
                    sourceSuffixes: ['styl', 'css', 'ie.styl', 'ie.css', 'ie9.styl', 'ie9.css'],
                    autoprefixer: { browsers: autoprefixer.getBrowsers(platform, 'ie9.css') }
                }],

                // Прогоняем через борщик
                [techs.borschik, {
                    source: '?.gemini.js',
                    target: '_?.gemini.js',
                    freeze: false,
                    minify: false
                }],
                [techs.borschik, {
                    source: '?.js',
                    target: '_?.js',
                    freeze: false,
                    minify: false
                }],
                [techs.borschik, {
                    source: '?.css',
                    target: '_?.css',
                    freeze: true,
                    minify: true,
                    tech: 'cleancss',
                    techOptions: {
                        cleancss: { advanced: false }
                    }
                }],
                [techs.borschik, {
                    source: '?.ie9.css',
                    target: '_?.ie9.css',
                    freeze: true,
                    minify: true,
                    tech: 'cleancss',
                    techOptions: {
                        compatibility: 'ie9',
                        cleancss: { advanced: false }
                    }
                }],
                [techs.smartCopy, {
                    sourceTargets: ['_?.css', '_?.ie9.css', '_?.js', '_?.gemini.js'],
                    dest: './gemini',
                    target: 'public'
                }]
            ]);

            nodeConfig.addTargets([
                '?.dirs',
                '?.files',
                'public',
                '_?.js',
                '_?.css',
                '_?.ie9.css',
                '?.priv.gemini.js',
                '_?.gemini.js'
            ]);
        });
    }
};
