BEM.DOM.decl('chart', {
    onSetMod: {
        js: {
            inited: function () {
                this._container = this.domElem.parent();
                this._tooltipBottom = this.findBlockOutside('b-page').findBlockInside({
                    blockName: 'popup',
                    modName: 'name',
                    modVal: 'dashboard-top'
                });
                this._tooltipTop = this.findBlockOutside('b-page').findBlockInside({
                    blockName: 'popup',
                    modName: 'name',
                    modVal: 'dashboard-bottom'
                });
                this._dateFormat = d3.timeFormat('%d.%m.%Y');
                this._canvasWidth = this._container.width();

                this._empty = false;

                this._type = this.params.type;
                this._prepareData();
                this._drawChart();
            }
        }
    },
    /* eslint-disable complexity */
    _Chart: function (opts, cb) {
        opts = opts || {};
        this.name = opts.name;
        this.padding = opts.padding || { top: 0, left: 0, right: 0 };
        this.width = opts.width;
        this.height = opts.height;
        this.widgetData = opts.data;
        this.currentData = opts.data;
        this.info = opts.info;
        this.animationDuration = opts.animationDuration || 0;
        this.tooltip = opts.tooltip || false;
        this.barWidth = opts.barWidth || false;
        this.hasCompare = opts.hasCompare || false;
        this.scale = 0;
        this.bars = {};
        this.OFFSET_COMPARE_BAR = 0.35;
        this.PARAMETERS = {
            'bar-item': {
                groupSelector: '#groupBar',
                widthRatio: 1,
                dataName: 'percent'
            },
            'bar-item-compare': {
                groupSelector: '#groupCompareBar',
                widthRatio: 0.3,
                dataName: 'compareData'
            }
        };

        this.x = d3.scaleTime().range([0, this.width - this.padding.left - this.padding.right]);
        this.y = d3.scaleLinear().range([this.height, 0]);

        cb.call(this);

        this.resize = function (width) {
            this.x.range([0, width - this.padding.left - this.padding.right]);
            this._updateCurrentBars(false);
        };

        this.addCanvas = function (canvas) {
            this.canvas = canvas
                .append('g')
                .attr('transform', 'translate(' + this.padding.left + ', ' + this.padding.top + ')')
                .attr('class', this.name);

            return this;
        };

        this.calculateScale = function () {
            this.scale = this.x(this.widgetData[1].date) - this.x(this.widgetData[0].date);

            return this.scale;
        };

        this._calculateWidthBar = function () {
            var fullWidth = this.calculateScale();

            this._widthBar = this.barWidth ? this.barWidth(fullWidth) : fullWidth;

            return this._widthBar;
        };

        this._getClassName = function (className, data) {
            className += data.percent < 0 ? ' neg' : ' pos';

            return className;
        };

        /**
         * Добавляет новые столбцы
         * @param {Object} selection Выборка d3
         * @param {Object} options Объект с опциями
         * @param {function|Number} options.calcX Значение или функция,
         *     возвращающая значение, для каждого X в выборке
         * @param {function|Number} options.calcY Значение или функция,
         *     возвращающая значение, для каждого Y в выборке
         * @param {String} options.className Название класса для каждого элемента в выборке
         * @returns {Object}
         * @private
         */
        this._addBar = function (selection, options) {
            var widthRatio = this.PARAMETERS[options.className].widthRatio;
            var type = this.PARAMETERS[options.className].dataName;

            return selection
                .append('rect')
                .attr('class', this._getClassName.bind(this, options.className))
                .attr('x', options.calcX)
                .attr('width', this._widthBar * widthRatio)
                .attr('y', options.calcY)
                .attr('height', 0)
                .on('mouseover', this._onMouseOverBar.bind(this, type))
                .on('mouseout', this._onMouseOutBar.bind(this));
        };

        /**
         * Удаляет столбцы из выборки
         * @param {Object} selection Выборка элементов d3, которые нужно удалить
         * @param {Boolean} deleteAll флаг удаления всех элементов
         * @returns {Object}
         * @private
         */
        this._deleteBar = function (selection, deleteAll) {
            var lastCurrentDate = this.currentData[this.currentData.length - 1].date;
            var startX = this.x(lastCurrentDate);
            var barsCounter = 0;

            return selection
                .transition()
                .duration(this.animationDuration)
                .attr('x', function (d) {
                    if (deleteAll) {
                        return this.x(d.date);
                    }
                    barsCounter += 1;

                    return startX + this._widthBar * barsCounter;
                }.bind(this))
                .attr('y', this.y(0))
                .attr('height', 0)
                .remove();
        };

        /**
         * Обновляет данные существующих столбцов
         * @param {Object} selection Выборка d3
         * @param {Object} options Объект с опциями
         * @param {String} options.className Название класса для каждого элемента в выборке
         * @param {Boolean} [options.hasAnimation = true] options.hasAnimation
         * @returns {Object}
         * @private
         */
        this._updateBar = function (selection, options) {
            var hasAnimation = options.hasAnimation !== false;
            var hasAnimationX = options.hasAnimationX !== false;
            var dataName = this.PARAMETERS[options.className].dataName;
            var widthRatio = this.PARAMETERS[options.className].widthRatio;

            this._selection = selection;
            this._selection.attr('class', this._getClassName.bind(this, options.className));

            if (hasAnimationX) {
                this
                    ._setTransition(hasAnimation ? this.animationDuration : 0)
                    ._updateXAndWidth(widthRatio);
            } else {
                this
                    ._updateXAndWidth(widthRatio)
                    ._setTransition(hasAnimation ? this.animationDuration : 0);
            }

            this._selection
                .attr('y', function (d) {
                    return this.y(Math.max(0, d[dataName]));
                }.bind(this))
                .attr('height', function (d) {
                    return Math.abs(this.y(0) - this.y(d[dataName]));
                }.bind(this));

            return selection;
        };

        this._setTransition = function (animationDuration) {
            this._selection = this._selection
                .transition()
                .duration(animationDuration);

            return this;
        };

        this._updateXAndWidth = function (widthRatio) {
            this._selection
                .attr('x', this._calculateXByDate.bind(this))
                .attr('width', this._widthBar * widthRatio);

            return this;
        };

        this._calculateXByDate = function (data) {
            return this.x(data.date);
        };

        /**
         * Обновляет текущий график
         * @param {Object} options Объект с опциями
         * @param {Array} options.data Новые данные
         * @param {Boolean} options.hasCompare Флаг сравнения
         * @param {Object} options.info Объект с данными для попапа
         */
        this.changeData = function (options) {
            this.currentData = options.data;
            this.hasCompare = options.hasCompare;
            this.info = options.info;

            this.redrawBars($.extend({}, options, {
                className: 'bar-item'
            }));

            this.calculateScale();
            var compareBar = this.canvas
                .selectAll('.bar-item-compare');

            if (!this.hasCompare) {
                this._deleteBar(compareBar, true);

                return;
            }

            if (!compareBar.empty()) {
                this.redrawBars($.extend({}, options, { className: 'bar-item-compare' }));

                return;
            }
            this.drawBars('bar-item-compare');
        };

        this._getCompareBarOffset = function () {
            var translate = this._widthBar * this.OFFSET_COMPARE_BAR;

            return 'translate(' + translate + ',0)';
        };

        /* eslint-disable max-params */
        this.drawBars = function (className, hasAnimation, hasAnimationX, y) {
            this._calculateWidthBar();
            this.bars[className] = this.canvas
                .select(this.PARAMETERS[className].groupSelector)
                .selectAll('.' + className)
                .data(this.currentData);

            var newBar = this._addBar(this.bars[className].enter(), {
                calcX: this._calculateXByDate.bind(this),
                calcY: y || this.y(0),
                className: className
            });

            newBar = this._setBarOffset(newBar, className);

            this.bars[className] = {
                enter: this._updateBar(newBar, {
                    className: className,
                    hasAnimation: hasAnimation,
                    hasAnimationX: hasAnimationX
                })
            };
        };

        this.redrawBars = function (options) {
            this._calculateWidthBar();
            var previousCountDates = this.canvas.selectAll('.' + options.className).size();

            this.bars[options.className] = this.canvas
                .select(this.PARAMETERS[options.className].groupSelector)
                .selectAll('.' + options.className)
                .data(this.currentData);

            // Высчитываем координаты по Х для каждого нового столбца в графике относительно
            // Предыдущей координаты по Х, прибавляя отступ от него
            var calculateXWithOffset = function (data) {
                var indexNewBar = this.currentData.indexOf(data) - previousCountDates;

                return options.xOfPreviousLastBar + (this._widthBar * (indexNewBar + 1));
            }.bind(this);

            var newBar = this._addBar(this.bars[options.className].enter(), {
                calcX: calculateXWithOffset,
                calcY: options.yOfPreviousLastBar,
                className: options.className
            });

            newBar = this._setBarOffset(newBar, options.className);

            this.bars[options.className].enter = this._updateBar(newBar, {
                className: options.className
            });

            this.bars[options.className].update = this._updateBar(this.bars[options.className], {
                className: options.className
            });

            this._deleteBar(this.bars[options.className].exit());
        };

        this._setBarOffset = function (selection, className) {
            if (className === 'bar-item-compare') {
                return selection.attr('transform', this._getCompareBarOffset());
            }

            return selection;
        };

        this.redrawChart = function (range, hasAnimation) {
            hasAnimation = hasAnimation !== false;
            this.x.domain(range);
            this._updateCurrentBars(hasAnimation);
        };

        this._updateCurrentBars = function (hasAnimation) {
            this._calculateWidthBar();

            if (this.hasCompare) {
                this._updateBar(this.canvas.selectAll('.bar-item-compare'), {
                    className: 'bar-item-compare',
                    hasAnimation: hasAnimation,
                    hasAnimationX: false
                });
            }

            this._updateBar(this.canvas.selectAll('.bar-item'), {
                className: 'bar-item',
                hasAnimation: hasAnimation,
                hasAnimationX: false
            });
        };

        this._onMouseOverBar = function (type, data) {
            if (!this.tooltip) {
                return;
            }

            var isCompared = type === 'compareData';
            var value = data[type];
            var tooltip = this.tooltip[value < 0 ? 'top' : 'bottom'];
            var info = this.info[isCompared ? 'compare' : 'main'];

            tooltip.setContent(BH.apply({
                block: 'chart-tooltip',
                mods: { type: 'dashboard' },
                data: {
                    title: info.title,
                    category: info.category,
                    info: info.other,
                    date: d3.timeFormat('%d.%m.%Y')(data.date),
                    percent: value.toFixed(2)
                }
            }));
            var $target = $(d3.event.target);
            var height = value < 0 ? $target.attr('height') : 0;
            var offsetHorizontal = Math.round($target.attr('width') / 2);

            tooltip.hide();
            tooltip.show($target.offset());
            tooltip.domElem.css({
                transform: 'translate(' + offsetHorizontal + 'px, ' + height + 'px)'
            });
        };

        this._onMouseOutBar = function () {
            if (!this.tooltip) {
                return;
            }
            this.tooltip.top.hide();
            this.tooltip.bottom.hide();
        };

        this.drawChart = function () {
            this.canvas.append('g')
                .attr('id', 'groupBar')
                .attr('clip-path', 'url(#rectClip)');

            this.canvas.append('g')
                .attr('id', 'groupCompareBar')
                .attr('clip-path', 'url(#rectClip)');

            this.drawBars('bar-item', true, false);

            return this;
        };

        return this;
    },

    _getInitialDateRange: function (data) {
        var length = data.length;
        var start = length <= this._TAIL ? data[0] : data[length - this._TAIL - 1];
        var end = data[length - 1];

        return [start.date, end.date];
    },

    _getCurrentRange: function (data) {
        /* eslint-disable max-len */
        var length = data.length;
        var range = d3.timeDays(this.freezedDateRange[0], this.freezedDateRange[1]).length;
        var start = this.freezedDateRange[0];
        var end = this.freezedDateRange[1];

        // Если первый день в периоде меньше первой даты в данных
        if (this.freezedDateRange[0] < data[0].date) {
            start = data[0].date;
            // Берем минимальный день из последней даты в данных и последней даты в новом отображаемом периоде
            // Чтобы правый край не выходил за границу данных
            end = d3.min([data[length - 1].date, d3.timeDay.offset(start, range)]);
        }

        // Если последний день в периоде больше последней даты в данных
        if (this.freezedDateRange[1] > data[length - 1].date) {
            end = data[length - 1].date;
            // Берем максимальный день из первой даты в данных и первой даты в новом отображаемом периоде
            // Чтобы правй край не выходил за границу данных
            start = d3.max([data[0].date, d3.timeDay.offset(end, -range)]);
        }

        return [start, end];
    },

    _getClipWidth: function (widthContainer) {
        return widthContainer - this._PADDING.left - this._PADDING.right - 4;
    },

    /**
     * Обработчик события изменения размеров окна
     * @private
     */
    _onWindowResize: function () {
        var newWidth = this._container.width();
        var newGraphWidth = newWidth - this._PADDING.left - this._PADDING.right;

        /* eslint-disable no-unused-expressions */
        this._canvas.select('.overlay').attr('width', newGraphWidth);
        this._yChartAxis.tickSize(newWidth);
        this._yMapAxis.tickSize(newGraphWidth);
        this._chart.resize(newWidth);
        this._map.resize(newWidth, 1);
        this._canvas.select('.clip-animate').attr('width', this._getClipWidth(newWidth));
        this._map.canvas.select('.y.axis').call(this._yMapAxis);
        this._chart.canvas.select('.y.axis').call(this._yChartAxis)
            .selectAll('text')
            .attr('x', -this._PADDING.left)
            .attr('dy', -4);

        this._map.width = newWidth;

        this._brush.remove();
        this._buildRange(this.currentData);

        this._brushRange = this._setRange();
        this._setSelectedBar(this._map.canvas.selectAll('.bar-item'), this.freezedDateRange);
    },

    _addClipPath: function () {
        this._canvas.append('clipPath')
            .attr('id', 'rectClip')
            .append('rect')
            .attr('class', 'clip-animate')
            .attr('width', 0)
            .attr('height', this._chart.height)
            .attr('x', 2)
            .attr('width', this._getClipWidth(this._canvasWidth));

        return this;
    },

    /**
     * Обработчик события изменения элемента brush
     * @private
     */
    _onBrush: function () {
        var range = d3.event.selection || this._getRange(this.freezedDateRange);
        var dateRange = this._getDateRange(range);
        var newDateRange = [];

        if (d3.event.type === 'brush') {
            newDateRange = dateRange.map(d3.timeDay.round);

            if (d3.timeDays(newDateRange[0], newDateRange[1]).length === this._TAIL) {
                this.freezedDateRange = newDateRange;
            }

            if (d3.timeDays(newDateRange[0], newDateRange[1]).length < this._TAIL) {
                newDateRange[0] = d3.min([newDateRange[0], this.freezedDateRange[0]]);
                newDateRange[1] = d3.max([newDateRange[1], this.freezedDateRange[1]]);
            }
        }

        if (!newDateRange.length) {
            newDateRange = dateRange;
        }

        this._chart.redrawChart(newDateRange, false);
        this._drawBrushLabels(range, newDateRange);
        this._chart.calculateScale();

        this._setSelectedBar(this._map.canvas.selectAll('.bar-item'), newDateRange);

        if (d3.event.sourceEvent && d3.event.sourceEvent.type !== 'brush') {
            this._moveBrush(this._getRange(newDateRange));
        }
    },

    _onBrushEnd: function () {
        var range = d3.event.selection;

        if (!range) {
            return;
        }
        this.freezedDateRange = this._getDateRange(range);
    },

    _moveBrush: function (range) {
        range = range || this._getRange(d3.brushSelection(this._brush.node()));
        this._map.canvas.select('.x.brush').call(this._brushRange.move, range);
    },

    _setSelectedBar: function (selection, period) {
        selection.classed('selected', function (d) {
            return d.date >= period[0] && d.date < period[1];
        });
    },

    _drawBrushLabels: function (range, dateRange) {
        var handles = this._map.canvas.selectAll('.handle--custom')
            .attr('transform', function (data, index) {
                return 'translate(' + range[index] + ', 0)';
            });

        handles.select('.text-from').text(this._dateFormat(dateRange[0]));
        handles.select('.text-to').text(this._dateFormat(dateRange[1]));
        this._map.canvas.selectAll('.handle')
            .attr('y', 0)
            .attr('width', 8)
            .attr('height', this._map.brushHeight);
    },

    _getRange: function (dateRange) {
        return [
            this._map.x(dateRange[0]),
            this._map.x(dateRange[1])
        ];
    },

    _getDateRange: function (coordinates) {
        return coordinates.map(this._map.x.invert, this._map.x);
    },

    _setRange: function () {
        return d3.brushX()
            .extent([
                [0, 0],
                [this._map.width - this._PADDING.left - this._PADDING.right, this._map.brushHeight]
            ])
            .on('start brush', this._onBrush.bind(this))
            .on('end', this._onBrushEnd.bind(this));
    },

    /**
     * Создает элемент для выбора диапазона
     * @param {Object} data
     * @returns {_buildRange}
     * @private
     */
    _buildRange: function (data) {
        var initialDateRange = this._getInitialDateRange(data);
        var initialRange = this._getRange(initialDateRange);

        this.freezedDateRange = initialDateRange;

        this._brushRange = this._setRange();

        this._brush = this._map.canvas.append('g')
            .attr('class', 'x brush')
            .style('opacity', 0)
            .call(this._brushRange)
            .call(this._brushRange.move, initialRange);

        var handles = this._brush.selectAll('.handle--custom')
            .data([
                { type: 'w' },
                { type: 'e' }
            ])
            .enter()
            .append('g')
            .attr('class', 'handle--custom');

        handles
            .append('text')
            .style('fill', 'black')
            .attr('class', function (_, index) {
                return index ? 'text-to' : 'text-from';
            });

        this._drawBrushLabels(initialRange, initialDateRange);

        this._brush.selectAll('rect')
            .attr('shape-rendering', 'crispEdges')
            .attr('height', this._map.brushHeight);

        this._brush
            .transition()
            .delay(this._ANIMATION_DURATION)
            .ease(d3.easeLinear)
            .duration(200)
            .style('opacity', 1);

        return this;
    }
});
