BEM.decl({
    block: 'chart2',
    modName: 'type',
    modVal: 'line'
}, {
    onSetMod: {
        js: {
            inited: function () {
                var desktopPadding = { top: 26, left: 40, right: 40 };
                var touchPadding = { top: 26, left: 30, right: 20 };

                this.TAIL = 7;
                this.CHART_OFFSET = this.isMobile ? 15 : 40;
                this.MAP_OFFSET = 360;
                this.MAP_MARGIN = 90;
                this.PADDING = this.isMobile ? touchPadding : desktopPadding;

                this._tooltip = {
                    top: this.findBlockOutside('b-page').findBlockInside({
                        blockName: 'popup',
                        modName: 'name',
                        modVal: 'chart-line-top'
                    })
                };

                this.__base.apply(this, arguments);
            }
        }
    },

    _prepareData: function () {
        var chart = this.params.chart;

        this.isCurve = chart.type === 'curve';
        this.data = {};
        this.axes = ['mainAxis'];

        this.axes.forEach(function (axis) {
            this.data[axis] = chart.data[axis].map(function (lineData) {
                lineData.values = lineData.values.map(function (dot) {
                    return {
                        date: d3.timeParse('%m.%y')(dot.date),
                        value: dot.value
                    };
                });

                return lineData;
            });
        }.bind(this));

        this.dates = this.data.mainAxis[0].values.map(function (dot) {
            return dot.date;
        });

        this.lines = {
            mainAxis: 'line',
            secondaryAxis: 'secondaryLine'
        };
    },

    _draw: function () {
        var self = this;
        var yRange = this._getYRange(this.data.mainAxis);

        this._chart = new this._Canvas({
            name: BEM.INTERNAL.buildClass('chart2', 'svg-chart'),
            width: this._canvasWidth,
            height: 318,
            data: this.data,
            tooltip: this._tooltip,
            postfix: this.params.chart.postfix
        }, function () {
            this.x.domain(d3.extent(self.dates));
            this.y.domain(yRange);

            self._addLine(this, self.isCurve);
        }, this);

        this._map = new this._Canvas({
            name: BEM.INTERNAL.buildClass('chart2', 'svg-map'),
            width: this._canvasWidth,
            height: 50,
            data: this.data
        }, function () {
            this.x.domain(self._chart.x.domain());
            this.y.domain(self._chart.y.domain());

            self._addLine(this, self.isCurve);
        }, this);

        this._svg = d3.select(this.domElem[0]).append('svg')
            .attr('width', this._canvasWidth)
            .attr('height', this._chart.height + this._map.height + this.MAP_MARGIN);

        this._chart.drawOn(this._svg);
        this._addClipPath();

        this._buildAxis();
        this._drawChartLines();

        this._map.drawOn(this._svg, { topOffset: this.MAP_OFFSET });
        this._drawMap();
        this._drawMapLines();
    },

    _addLine: function (context, isCurve) {
        context.line = d3.line()
            .x(context.accessor.bind(context, 'x'))
            .y(context.accessor.bind(context, 'y'));

        if (isCurve) {
            context.line = context.line.curve(d3.curveMonotoneX);
        }
    },

    _getYHandler: function (axis, context) {
        if (axis === 'mainAxis') {
            return context.accessor.bind(context, 'y');
        } else if (axis === 'secondaryAxis') {
            return function (d) {
                return context.y2(d.value);
            };
        }
    },

    /**
     * Перерисовывает данные на графике при ресайзе страницы
     * @private
     */
    /* eslint-disable no-empty-function */
    _updateChartData: function () {},

    _buildAxis: function () {
        this._yChartAxis = d3.axisRight()
            .scale(this._chart.y)
            .ticks(6)
            .tickSize(this._chart.width);

        this._chart.canvasSvg.append('g')
            .attr('class', 'y axis chart')
            .call(this._yChartAxis)
            .selectAll('text')
            .attr('x', -this._chart.padding.left);

        var ticks = this._chart.canvasSvg.selectAll('.axis.chart .tick line');

        ticks.each(function (data) {
            var tick = d3.select(this);

            tick.classed('bold', function () {
                return data === 0;
            });
        });

        this._xChartAxis = d3.axisBottom()
            .scale(this._chart.x)
            .tickSize(0)
            .tickFormat(function (d) {
                return d3.timeFormat('%m.%y')(d);
            });

        if (this.isMobile) {
            this._xChartAxis = this._xChartAxis.ticks(d3.timeMonth.every(2));
        }

        this._chart.canvasSvg.append('g')
            .attr('transform', 'translate(0,' + this._chart.height + ')')
            .attr('class', 'axis--x')
            .call(this._xChartAxis)
            .selectAll('text')
            .attr('dy', this._chart.padding.top / 2);
    },

    _drawChartLines: function () {
        this.axes.forEach(function (axis) {
            this.data[axis].forEach(function (lineData) {
                var line = this.lines[axis];
                var dots = this._getChartDrawDots(lineData);

                this._chart.canvasSvg.append('path')
                    .attr('clip-path', 'url(#rectClip)')
                    .datum(lineData.values)
                    .attr('class', 'line')
                    .style('stroke', lineData.color)
                    .attr('d', this._chart[line]);

                this._chart.canvasSvg
                    .selectAll('dot')
                    .data(dots)
                    .enter()
                    .append('circle')
                    .attr('class', 'dot')
                    .attr('clip-path', 'url(#dotClip)')
                    .attr('r', 4)
                    .attr('cx', this._chart.accessor.bind(this._chart, 'x'))
                    .attr('cy', this._getYHandler(axis, this._chart))
                    .style('fill', lineData.color)
                    .on('mouseover', this._chart._onMouseOver.bind(this._chart, 'line'))
                    .on('mouseout', this._chart._onMouseOut.bind(this._chart));
            }.bind(this));
        }.bind(this));
    },

    _getChartDrawDots: function (lineData) {
        if (this.isCurve) {
            return [lineData.values[lineData.values.length - 1]];
        }

        return lineData.values;
    },

    /* eslint-disable complexity */
    _onBrushMove: function () {
        var selection = d3.event.selection;
        var dateRange = this._getDateRange(selection);
        var newDateRange = dateRange;

        if (d3.event.type === 'brush') {
            newDateRange = dateRange.map(d3.timeMonth.round);
            var timeMonthsLength = d3.timeMonths(newDateRange[0], newDateRange[1]).length;

            if (timeMonthsLength === this.TAIL) {
                this.freezedDateRange = newDateRange;
            }

            if (timeMonthsLength < this.TAIL) {
                newDateRange[0] = d3.min([newDateRange[0], this.freezedDateRange[0]]);
                newDateRange[1] = d3.max([newDateRange[1], this.freezedDateRange[1]]);
            }
        }

        if (d3.event.sourceEvent && d3.event.sourceEvent.type !== 'brush') {
            this._moveBrush(this._getRange(newDateRange));
        }

        this._chart.x.domain(newDateRange);
        this._chart.canvasSvg
            .selectAll('.axis--x')
            .call(this._xChartAxis)
            .selectAll('text')
            .attr('dy', this._chart.padding.top / 2);

        this.axes.forEach(function (axis) {
            var line = this.lines[axis];

            this._chart.canvasSvg.selectAll('.line').attr('d', this._chart[line]);

            this._chart.canvasSvg
                .selectAll('.dot')
                .attr('cx', this._chart.accessor.bind(this._chart, 'x'))
                .attr('cy', this._getYHandler(axis, this._chart));
        }.bind(this));

        this._drawBrushLabels();
    },
    /* eslint-enable complexity */

    _drawMapLines: function () {
        this.axes.forEach(function (axis) {
            var line = this.lines[axis];

            this.data[axis].forEach(function (lineData) {
                var lastDot = lineData.values[lineData.values.length - 1];

                this._map.canvasSvg.append('path')
                    .datum(lineData.values)
                    .attr('class', 'line')
                    .style('stroke', lineData.color)
                    .attr('d', this._map[line]);

                this._map.canvasSvg
                    .selectAll('dot')
                    .data([lastDot])
                    .enter()
                    .append('circle')
                    .attr('r', 3)
                    .attr('cx', this._map.accessor.bind(this._map, 'x'))
                    .attr('cy', this._getYHandler(axis, this._map))
                    .style('fill', lineData.color);
            }.bind(this));
        }.bind(this));
    },

    _addClipPath: function () {
        this._appendDefs({
            id: 'rectClip',
            offset: 0
        });

        this._appendDefs({
            id: 'dotClip',
            offset: 5
        });

        this._svg.append('g')
            .attr('id', 'groupLines')
            .attr('clip-path', 'url(#rectClip)');

        this._svg.append('g')
            .attr('id', 'groupDots')
            .attr('clip-path', 'url(#dotClip)');
    },

    _appendDefs: function (data) {
        /* eslint-disable max-len */
        var width = this._chart.width - (2 * this.CHART_OFFSET) - this._chart.padding.left + (2 * data.offset);

        this._chart.canvasSvg.append('defs')
            .append('clipPath')
            .attr('id', data.id)
            .append('rect')
            .attr('class', 'clip-animate')
            .attr('y', 0)
            .attr('height', this._chart.height)
            .attr('x', this.CHART_OFFSET - data.offset)
            .attr('width', width);
    }
});
