BEM.DOM.decl('i-bem', {}, {

    findById: function (collection, predicate) {
        for (var i = 0, length = collection.length; i < length; i += 1) {
            var item = collection[i];
            var keys = Object.keys(predicate);
            var detected = keys.every(function (key) {
                return this[key] === predicate[key];
            }, item);

            if (detected) {
                return item;
            }
        }

        return null;
    },

    changeUrl: function (data, title, url) {
        if (window.history && history.pushState) {
            window.history.pushState(data, title, url);

            return;
        }
        window.location.href = url;
    },

    replaceUrl: function (data, title, url) {
        if (window.history && history.replaceState) {
            window.history.replaceState(data, title, url);
        }
    },

    pluck: function (collection, property) {
        var result = [];

        for (var i = 0, length = collection.length; i < length; i += 1) {
            result.push(collection[i][property]);
        }

        return result;
    },

    buildSelector: function () {
        var args = Array.prototype.slice.call(arguments);

        return '.' + BEM.INTERNAL.buildClass.apply(null, args);
    },

    decodeQuery: function (query) {
        return decodeURIComponent(query.substring(1))
            .split('&')
            .reduce(function (result, sURLVariable) {
                var sParameterName = sURLVariable.split('=');

                if (sParameterName[0] && sParameterName[1]) {
                    result[sParameterName[0]] = sParameterName[1];
                }

                return result;
            }, {});
    },

    /**
     * Возвращает пересечение двух массивов
     * @param {Array} thisArray
     * @param {Array} thatArray
     * @returns {Array}
     */
    getIntersection: function (thisArray, thatArray) {
        return thisArray.reduce(function (acc, thisItem) {
            if (thatArray.indexOf(thisItem) > -1) {
                acc.push(thisItem);
            }

            return acc;
        }, []);
    },

    /**
     * Web API для работы с LocalStorage
     * @returns {{get: get, set: set, remove: remove, clear: clear}}
     */
    getStorage: function () {
        var storage = window.localStorage;

        return {

            /**
             * Получение значения из Storage по ключу
             * @param {String} key
             * @returns {String | null}
             */
            get: function (key) {
                try {
                    return storage && storage.getItem(key);
                } catch (e) {
                    return null;
                }
            },

            /**
             * Сохранить ключ и значение в Storage
             * @param {String} key
             * @param {String} value
             */
            set: function (key, value) {
                if (storage) {
                    try {
                        storage.setItem(key, value);
                    } catch (e) { }
                }
            },

            /**
             * Удалить ключ из Storage
             * @param {String} key
             */
            remove: function (key) {
                if (storage) {
                    try {
                        storage.removeItem(key);
                    } catch (e) { }
                }
            },

            /**
             * Очистить Storage
             */
            clear: function () {
                if (storage) {
                    try {
                        storage.clear();
                    } catch (e) { }
                }
            }
        };
    },

    /**
     * Скейл айфреймов в мобильный версии
     * @param {Object} context
     */
    touchIframeFix: function (context) {
        if (!BH.lib.global.isMobile) {
            return;
        }

        var iframes = $(context.domElem.find('iframe'))
            .filter( function (_, iframe) {
                var iframeType = iframe.getAttribute('data-iframe-type');

                return ['chat', 'form'].indexOf(iframeType) === -1;
            });

        // Оборачиваем все iframe в div
        $.each(iframes, function (_, iframe) {
            var $wrapper = $('<div class="touch-iframe-wrapper"></div>');

            var $iframe = $(iframe);

            $iframe.css('transform-origin', 'top left');
            $iframe.wrap($wrapper);
        });

        // Перезапускаем масштабирование при изменении размеров экрана и смене ориентации
        context.bindTo($(window), 'orientationchange resize',
            this._scaleIframes.bind(this, iframes));

        this._scaleIframes(iframes);
    },

    _scaleIframes: function (iframes) {
        iframes.each(function (_, iframe) {
            var $iframe = $(iframe);
            var $wrapper = $iframe.parent();
            var wrapperParentWidth = this._getVisibleParentWidth($wrapper);
            var iframeWidth = $iframe.outerWidth();

            var scale = 1;
            var width = 'auto';
            var height = 'auto';

            // Если iframe шире чем viewport, вычисляем коэффициент масштабирования
            if (wrapperParentWidth < iframeWidth) {
                scale = wrapperParentWidth / iframeWidth;
                width = wrapperParentWidth;
                height = $iframe.outerHeight() * scale;
            }

            // Переносим все отступы с iframe на wrapper
            if ($iframe.innerHeight() > $iframe.height()) {
                $wrapper.css('padding', $iframe.css('padding'));
                $iframe.css('padding', '');
            }

            if ($iframe.css('border')) {
                $iframe.css('border', '');
                $iframe.attr('frameborder', 0);
            }

            if ($iframe.css('margin')) {
                $wrapper.css('margin', $iframe.css('margin'));
                $iframe.css('margin', 0);
            }

            $iframe.css('transform', 'scale(' + scale + ')');
            $wrapper.css({
                width: width,
                height: height
            });
        }.bind(this));
    },

    _getVisibleParentWidth: function (element) {
        var parents = element.parents();
        var allGaps = 0;
        var parentWidth = 0;

        for (var i = 0; i < parents.length; i = i + 1) {
            var parent = parents[i];
            var isVisible = parent.offsetWidth > 0;
            var parentGap = this._getElementGap($(parent), { withMargin: !isVisible });

            if (isVisible) {
                parentWidth = parent.offsetWidth - parentGap;
                break;
            }

            allGaps = allGaps + parentGap;
        }

        return parentWidth - allGaps;
    },

    _getElementGap: function (parent, params) {
        var paddingLeft = this._getGap(parent, 'paddingLeft');
        var paddingRight = this._getGap(parent, 'paddingRight');
        var marginLeft = this._getGap(parent, 'marginLeft');
        var marginRight = this._getGap(parent, 'marginRight');

        var paddings = paddingLeft + paddingRight;
        var margins = params && params.withMargin ? marginLeft + marginRight : 0;

        return paddings + margins;
    },

    _getGap: function (parent, gapName) {
        var gapNumber = Number(parent.css(gapName).replace('px', ''));

        if (!gapNumber) {
            return 0;
        }

        return gapNumber;
    },

    /**
     * Скролл до 60% страницы
     * @param {String} goal
     * @param {Object} handler
     * @returns {Boolean}
     */
    detectScrollMiddle: function (goal, handler) {
        var winBottomOffset = BEM.DOM.win.scrollTop() + BEM.DOM.win.height();
        var isMiddleReached = winBottomOffset > BEM.DOM.doc.height() * 0.6;

        if (isMiddleReached) {
            if (goal) {
                BEM.blocks.metrika.reachGoal(goal);
            }

            if (handler) {
                this.unbindFromWin('scroll', handler);
            }
        }

        return isMiddleReached;
    },

    /**
     * Скролл до конца элемента
     * @param {Object} $elem
     * @param {String} goal
     * @param {Function} handler - функция для анбиндинга
     * @param {number} threshold - с какой части элемент считает увиденным.
     * от 0 до 1, при этом 0 - верхняя граница элемента, 1 - нижняя
     * @returns {Boolean}
     */
    // eslint-disable-next-line max-params
    detectScrollEnd: function ($elem, goal, handler, threshold) {
        threshold = isNaN(threshold) ? 1 : threshold;

        var winBottomOffset = BEM.DOM.win.scrollTop() + BEM.DOM.win.height();
        var elemBottomOffset = $elem.offset().top + $elem.height() * threshold;
        var isEndReached = winBottomOffset > elemBottomOffset;

        if (isEndReached) {
            if (goal) {
                BEM.blocks.metrika.reachGoal(goal);
            }

            if (handler) {
                this.unbindFromWin('scroll', handler);
            }
        }

        return isEndReached;
    },

    /**
     * Проверяет, не выходит ли элемент за границу экрана
     * @param {Object} $elem
     * @param {String} direction, граница, за которую вышел элемент (top|bottom|both)
     * @param {Number} offset - доп смещение
     * @returns {Boolean}
     */
    checkOnScreen: function ($elem, direction, offset) {
        direction = direction || 'top';
        offset = offset || 0;

        var topThreshold = BEM.DOM.win.scrollTop() + offset;
        var bottomThreshold = BEM.DOM.win.scrollTop() + BEM.DOM.win.height() - offset;

        var elemTop = $elem.offset().top;
        var elemBottom = $elem.offset().top + $elem.height();

        var itemOnScreen = (
            (direction === 'top' && elemBottom > topThreshold) ||
            (direction === 'bottom' && elemTop < bottomThreshold) ||
            (direction === 'both' && elemBottom > topThreshold && elemTop < bottomThreshold)
        );

        return itemOnScreen;
    },

    /**
     * Функция реализует предзагрузку изображений
     * @param {String []} images
     */
    preloadImages: function (images) {
        if (!images || images.length === 0) {
            return;
        }

        images.forEach(function (src) {
            BEM.DOM.append(document.body, BH.apply({
                tag: 'img',
                attrs: {
                    src: src,
                    style: 'display: none;'
                }
            }));
        });
    },

    /**
     * Меняет URL текущей страницы
     * @param {String} url
     */
    changeCurrentUrl: function (url) {
        if (!window.history || typeof window.history.replaceState !== 'function') {
            return;
        }

        var state = window.history.state;

        window.history.replaceState(state, state && state.pageName, url);
    },

    _isSetTime: false,

    /**
     * Отправляет данные в метрику при достижении заданного числа милисекунд
     * @param {String} goalId
     * @param {Number|*} timeout
     */
    reachTime: function (goalId, timeout) {
        timeout = timeout || 30000;

        if (!goalId || this._isSetTime) {
            return;
        }

        this._isSetTime = true;

        setTimeout(function () {
            BEM.blocks.metrika.reachGoal(goalId);
        }, timeout);
    },

    /**
     * Убираем полсдений слэш из урла
     * @param {String} str
     * @returns {String}
     */
    removeLastSlash: function(str) {
        if (typeof str !== 'string') {
            return str;
        }

        return str.replace(/\/$/, '');
    }
});
