module.exports = function (bh) {

    /**
     * Формирует элемент выпадающего списка
     * @param {Object} item
     * @returns {{val, text}}
     */
    function getItem(item) {
        return {
            val: item.slug,
            text: item.name
        };
    }

    /**
     * Формирует набор элементов списка
     * @param {Object []} items
     * @returns {Object}
     */
    function getItems(items) {
        var notCategorizedItems = [
            getItem({
                slug: 'all',
                name: bh.lib.i18n('solutions', 'clear')
            })
        ];
        var groups = items.reduce(function (acc, item) {
            if (!item.group) {
                notCategorizedItems.push(getItem(item));
            } else if (acc.indexOf(item.group) < 0) {
                acc.push(item.group);
            }

            return acc;
        }, []);

        return notCategorizedItems.concat(groups
            .map(function (group) {
                return {
                    title: group,
                    items: items
                        .filter(function (item) {
                            return item.group === group;
                        })
                        .map(function (item) {
                            return getItem(item);
                        })
                };
            }));
    }

    /**
     * Возвращает представление выпадающего списка
     * @param {String} filterCategory
     * @param {Object []} items
     * @param {Object} filter
     * @returns {Object}
     */
    function getSelect(filterCategory, items, filter) {
        var isValueAll = filter.value === 'all';
        var isRequiredCategory = filter.by === filterCategory;

        return items && items.length && {
            block: 'select2',
            mods: {
                theme: 'normal',
                type: 'radiocheck',
                text: 'vary',
                size: 'm',
                width: 'max',
                category: filterCategory
            },
            val: !isValueAll && isRequiredCategory && filter.value,
            text: bh.lib.i18n('solutions', filterCategory),
            items: getItems(items),
            popup: {
                mix: {
                    block: 'popup2',
                    mods: { order: 'top' }
                },
                directions: ['bottom-left']
            }
        };
    }

    /**
     * Возвращает изображение для данного идентификатора региона
     * @param {Object} page
     * @param {Number} regionId
     * @returns {String}
     */
    function getGeoImage(page, regionId) {
        var geoObjects = page.geoImages.filter(function (geoImage) {
            return geoImage.geoId === regionId;
        });

        return (geoObjects.length > 0 && geoObjects[0].image) || page.backgroundImage;
    }

    /**
     * Возвращает фоновое изображение для этажа решений
     * @param {String} industry
     * @param {Object} page
     * @param {Number} regionId
     * @returns {String}
     */
    function getImage(industry, page, regionId) {
        return (industry && industry.image) || getGeoImage(page, regionId);
    }

    /* eslint-disable complexity */
    bh.match('level-solution', function (ctx, json) {
        var isMobile = bh.lib.global.isMobile;
        var page = json.page;
        var align = page.align || 'left';
        var color = page.color || 'black';
        var industrySlug = json.industry;
        var industry = bh.lib.findIndustry(page.industries, industrySlug);
        var image = getImage(industry, page, json.regionId);
        var filter = json.filter;
        var filters = {
            tasks: page.tasks,
            industries: page.industries,
            products: page.products,
            companySize: page.companySize,
            filterUrl: json.filterUrl,
            section: json.section,
            defaultFilter: filter,
            solutionsUrl: json.solutionsUrl,
            industry: industrySlug,
            defaultImage: getGeoImage(page, json.regionId),
            currentImage: image
        };

        ctx
            .js(filters)
            .mods({
                align: align,
                image: image && 'yes',
                color: color
            })
            .attrs({ style: !isMobile && image && 'background-image: url("' + image + '");' })
            .content([
                {
                    tag: 'h1',
                    elem: 'title',
                    content: page.title
                },
                {
                    elem: 'text',
                    mix: { block: 'html-content' },
                    content: page.text
                },
                Object.assign({
                    elem: 'filters',
                    filter: filter
                }, filters),
                { block: 'notifications-list' }
            ]);
    });
    /* eslint-enable complexity */

    bh.match('level-solution__filters', function (ctx, json) {
        var filter = json.filter;

        var controls = [
            getSelect('tasks', json.tasks, filter),
            getSelect('industries', json.industries, filter),
            getSelect('products', json.products, filter),
            getSelect('companySize', json.companySize, filter)
        ];

        ctx.content(controls.map(function (control) {
            return control ? {
                elem: 'filter',
                content: control
            } : null;
        }));
    });
};
