module.exports = function (bh) {
    'use strict';

    /**
     * Поддержка форматов "Этаж - Журнал" и "Блок трансляции и видео"
     * @param {Object} level
     * @returns {Array}
     */
    function getLinks(level) {
        if (level.links) { // Этаж - Журнал
            return level.links;
        }

        if (level.columns) { // Блок трансляции и видео
            return level.columns.reduce(function (acc, column) {
                return acc.concat(column);
            }, []);
        }

        return []; // ¯\_(ツ)_/¯
    }

    /* eslint-disable complexity */
    bh.match('level_type_journal', function (ctx, json) {
        if (!json.level) {
            return '';
        }

        var level = json.level;
        var isIndexPage = json.page === 'index';
        var hasTimer = Boolean(level.translationDate);
        var translationType = level.translationType;
        var isTranslation = !hasTimer && translationType && translationType !== 'no';
        var fields = level['subscription-fields'];
        var isSubscription = fields && fields.length > 0;
        var links = getLinks(level);
        var hasContent = links.length > 0;
        var goals = level.metrika || {};
        var shareUrl = json.journalsPage + '?journal=' + level.anchor + '#' + level.anchor;

        if (isIndexPage) {
            links = links.slice(0, 6);
        }

        var journalContent = links.map(function (item) {
            return {
                block: 'level-card',
                mods: {
                    type: 'journal'
                },
                linkColor: 'white',
                link: {
                    prefix: json.newsPage,
                    url: item.url,
                    slug: item.link,
                    text: item.text,
                    image: item.image
                },
                date: item.date,
                commentsData: bh.lib.util.getCommentsCount(json.commentsData, item.link),
                commentsCounterType: 'dark'
            };
        });

        var shareElem = {
            elem: 'share',
            title: level.shareTitle,
            og: level.og,
            metrikaGoalId: goals.share,
            url: shareUrl
        };

        var isShareInContent = !isIndexPage && !bh.lib.global.isTouch;

        if (isShareInContent) {
            journalContent.push(shareElem);
        }

        ctx
            .js({
                metrikaGoalId: goals.show,
                shareUrl: shareUrl,
                og: level.og
            })
            .attrs({ id: level.anchor })
            .mods({
                player: isTranslation && 'yes',
                subscription: isSubscription,
                timer: hasTimer && 'yes',
                page: json.page
            })
            .content([
                {
                    elem: 'journal',
                    mods: {
                        type: isSubscription && 'subscription',
                        content: hasContent ? 'yes' : 'no'
                    },
                    level: level,
                    transactionPath: json.transactionPath,
                    isSubscription: isSubscription,
                    isTranslation: isTranslation,
                    hasTimer: hasTimer,
                    browserName: json.browserName,
                    fields: fields,
                    isBrowserValid: json.isBrowserValid,
                    comdiUrl: json.comdiUrl,
                    page: json.page
                },
                hasContent && {
                    elem: 'content',
                    elemMods: { type: 'journal' },
                    columns: journalContent
                },
                !isShareInContent && !hasTimer && {
                    elem: 'share',
                    title: level.shareTitle,
                    og: level.og,
                    metrikaGoalId: goals.share,
                    url: shareUrl
                }
            ]);
    });

    bh.match('level__journal', function (ctx, json) {
        var level = json.level || {};
        var chatUrl = level.chatUrl;
        var isSubscription = json.isSubscription;
        var hasTimer = json.hasTimer;
        var image = ctx.tParam('mdsImage');

        if (json.isTranslation) {
            return [
                {
                    elem: 'player',
                    mods: { type: level.translationType },
                    level: level,
                    isBrowserValid: json.isBrowserValid,
                    comdiUrl: json.comdiUrl
                },
                chatUrl && {
                    elem: 'chat',
                    title: level.chatTitle,
                    url: chatUrl
                }
            ];
        }

        var translationDate = null;

        if (level.translationDate) {
            // UTC+3 для МСК
            translationDate = bh.lib.util.getUTCDate(level.translationDate, 3).toISOString();
        }

        ctx
            .mods({
                type: isSubscription && 'subscription',
                animated: json.browserName !== 'MSIE' && !hasTimer && json.page === 'index',
                timer: hasTimer && 'yes'
            })
            .tag(isSubscription || hasTimer ? 'div' : 'a')
            .attrs({
                href: level.imageLink
            })
            .content([
                translationDate && {
                    elem: 'journal-timer',
                    content: {
                        block: 'timer',
                        js: {
                            endTime: translationDate
                        }
                    }
                },
                image && {
                    elem: 'journal-image',
                    content: {
                        tag: 'img',
                        attrs: { src: image }
                    }
                },
                {
                    elem: 'journal-text',
                    content: level.text
                },
                {
                    elem: 'journal-description',
                    mix: { block: 'html-content' },
                    content: level.description
                },
                translationDate && {
                    elem: 'journal-form',
                    content: {
                        block: 'adaptive-form',
                        data: {
                            url: level.formUrl,
                            attrs: {
                                name: 'journal-form',
                                width: '554px',
                                height: bh.lib.global.isMobile ? '263px' : '108px'
                            },
                            theme: 'vda-journal',
                            params: {
                                name: 'journal-form',
                                adaptive: true
                            }
                        }
                    }
                },
                isSubscription && {
                    elem: 'journal-form',
                    content: {
                        block: 'contact-form',
                        mods: { type: 'compact', theme: 'dark' },
                        js: {
                            subscriptionType: 'public',
                            url: json.transactionPath,
                            metrikaGoalId: 'SUBSCRIBE',
                            enableMetrika: json.enableMetrika,
                            extraAnalytics: true,
                            formName: 'journal-form'
                        },
                        fields: json.fields,
                        agreements: level.agreements,
                        button: {
                            text: bh.lib.i18n('forms', 'subscription.button.register'),
                            align: 'center'
                        },
                        successTitle: bh.lib.i18n('forms', 'subscribed.single.title'),
                        successText: bh.lib.i18n('forms', 'subscribed.single.text')
                    }
                }
            ]);
    });
    /* eslint-enable complexity */

    bh.match('level_type_journal__share', function (ctx, json) {
        var title = json.title;
        var og = json.og || {};
        var hasOg = og.title || og.description || og.image;

        if (!title && !hasOg) {
            return '';
        }

        ctx.content([
            {
                elem: 'share-title',
                mix: { block: 'html-content' },
                content: title
            },
            hasOg && {
                elem: 'share-social',
                content: {
                    block: 'share',
                    mods: { size: 'm' },
                    metrikaGoalId: json.metrikaGoalId,
                    url: json.url,
                    ogData: og
                }
            }
        ]);
    });

    bh.match('level__player_type_comdi', function (ctx, json) {
        ctx.content(json.isBrowserValid ? {
            block: 'comdi-player',
            event: json.level['video-event'],
            comdiUrl: json.comdiUrl
        } : { elem: 'not-supported' });
    });

    bh.match('level__player_type_youtube', function (ctx, json) {
        ctx.content({
            elem: 'player-video',
            content: {
                tag: 'iframe',
                attrs: {
                    frameborder: 0,
                    allowfullscreen: true,
                    width: 962,
                    height: 541,
                    src: bh.lib.util.placeholder(
                        'https://www.youtube-nocookie.com/embed/{s}?rel=0&amp;showinfo=0',
                        json.level.youtubeId
                    )
                }
            }
        });
    });

    bh.match('level__player_type_yaVideo', function (ctx, json) {
        var url = json.level.yaVideo;

        if (!url) {
            return '';
        }

        ctx.content({
            tag: 'iframe',
            attrs: {
                src: url,
                width: 966,
                height: 545,
                frameborder: 0,
                scrolling: 'no',
                allowfullscreen: 1
            }
        });
    });

    bh.match('level__not-supported-description', function (ctx) {
        ctx
            .tag('h1')
            .mix({ block: 'html-content' })
            .content(bh.lib.i18n('common', 'comdi.not-supported'));
    });

    bh.match('level__chat', function (ctx, json) {
        var title = json.title;
        var url = json.url;

        ctx.content([
            {
                elem: 'chat-header',
                content: [
                    {
                        tag: 'h3',
                        elem: 'chat-title',
                        content: title
                    },
                    {
                        tag: 'button',
                        elem: 'chat-button'
                    }
                ]
            },
            {
                elem: 'chat-content',
                content: {
                    tag: 'iframe',
                    attrs: {
                        'data-iframe-type': 'chat',
                        frameborder: 0,
                        allowfullscreen: false,
                        src: url
                    }
                }
            }
        ]);
    });
};
