const AWS = require('aws-sdk');
const fs = require('fs');
const path = require('path');

const UploadDir = function(options) {
    const {
        region = 'us-east-1',
        endpoint,
        accessKeyId,
        secretAccessKey,
        bucketName,
    } = options;

    const s3 = new AWS.S3({
        endpoint,
        region,
        secretAccessKey,
        accessKeyId,
        httpOptions: {
            timeout: 10000
        }
    });

    const getContentTypeByFileName = filePath => {
        switch (path.extname(filePath)) {
            case '.js':
                return 'application/x-javascript';
            case '.css':
                return 'text/css';
            case '.png':
                return 'image/png';
            case '.jpg':
                return 'image/jpeg';
            case '.svg':
                return 'image/svg+xml';
            case '.html':
                return 'text/html';
            default:
                return 'application/octet-stream';
        }
    };

    function putFile(filePath, s3Path) {
        const params = {
            Bucket: bucketName,
            Key: s3Path,
            Body: fs.readFileSync(filePath),
            ContentType: getContentTypeByFileName(filePath),
            ContentLanguage: 'html'
        };

        return new Promise(function(onFulfill, onReject) {
            s3.putObject(params, function(err) {
                if (err) {
                    console.log('uploaded err', err);

                    onReject(err);
                }
                onFulfill(filePath);
            });
        });
    }

    /**
     * Рекурсивный обход дирректории
     * если по текущее директории лежит папка,
     * @param {currentDirPath} путь до папки или файла
     * @param {Function} callback - функция-обработчик
     * @param {currentRelPath} текущий относительный путь
     */
    function walkSync(currentDirPath, callback, currentRelPath = '') {
        fs.readdirSync(currentDirPath).forEach(function (name) {
            var filePath = path.join(currentDirPath, name);
            var relativePath = path.join(currentRelPath, name);

            var stat = fs.statSync(filePath);

            if (stat.isFile()) {
                callback(filePath, stat, relativePath);
            } else if (stat.isDirectory()) {
                walkSync(filePath, callback, relativePath);
            }
        });
    }


    /**
     * Опубликовать папку на s3
     * @param {String} folder - путь до папки
     * @param {Function} pathBuilder - функция построения пути в s3
     */
    return function (folder, pathBuilder) {
        const promisses = []

        walkSync(folder, function(filePath, stat, relativePath) {
            const s3Path = pathBuilder(folder, relativePath);
            promisses.push( putFile(filePath, s3Path) );
        });

        return Promise.all(promisses)
            .then(() => {
                return folder;
            });
    }
};

module.exports = UploadDir;
