import url from 'url';

import React, { ChangeEvent, Fragment, useEffect, useRef, useState } from 'react';
import { useDispatch } from 'react-redux';

import FormattedContent, { IFormattedContentNodes } from 'client/common/components/formatted-content';
import RegionSwitcher from 'client/common/components/region-switcher';

import { useAgenciesFilters, useComponents, useLegoComponents, useToggle } from 'client/common/hooks';

import {
    addMultiFilter,
    changeSingleFilter,
    removeMultiFilter,
    replaceMultiFilter,
    setBudget,
    setBudgetOrder
} from 'store/agencies/actions';

import {
    IAgenciesFilter,
    TBudget,
    TFilterId,
    TMultiValueFilterId,
    TSingleValueFilterId
} from 'store/agencies/types';

import cn from 'utils/cn';
import { bindKeyset } from 'utils/i18n';
import { reachGoal } from 'utils/metrika';

import { CertificateType, IBudgetConfig, TRegion } from 'client/common/types';

import { IFilter, IFilterOption, TFilterGoals } from './types';

import './index.css';

const b = cn('agencies-filters');
const i18n = bindKeyset('agencies');

const filterGoals: TFilterGoals = {
    'city': {
        'common': 'GOROD'
    },
    'certificate': {
        'options': {
            'direct': 'DIREKT',
            'market': 'MARCET',
            'metrika': 'METRICA'
        }
    }
};

function reachFilterGoal(filterId: TFilterId, optionId: string) {
    const commonGoal = filterGoals[filterId]?.common;
    const optionGoal = filterGoals[filterId]?.options?.[optionId];

    if (commonGoal) {
        reachGoal(commonGoal);
    }

    if (optionGoal) {
        reachGoal(optionGoal);
    }
}

interface IAgenciesFiltersProps {
    filtersTemplate: string;
    description: string;
    regions: TRegion[];
    filters: IFilter[];
    tld: string;
    budgetConfig: IBudgetConfig;
    className?: string;
}

function findOptionById(options: IFilterOption[], id: string): IFilterOption {
    return options.find(option => option.id === id)!;
}

function AgenciesFilters(props: IAgenciesFiltersProps) {
    const {
        className = '',
        tld,
        regions,
        filtersTemplate,
        description,
        filters,
        budgetConfig
    } = props;

    const checkedFilters = useAgenciesFilters();

    const isDirectChecked = Array.isArray(checkedFilters.certificate) ?
        checkedFilters.certificate.includes(CertificateType.Direct) :
        checkedFilters.certificate === CertificateType.Direct;

    const hasBudgetFilter = budgetConfig.hasFilter && isDirectChecked;

    const listService = filters.filter(item => item.id === 'service');
    const services = listService.length ? listService[0].options : [];
    const filterNodes = filters.reduce((acc: IFormattedContentNodes, filter) => {
        acc[filter.id] = (
            <FiltersGroup
                {...filter}
                />
        );

        return acc;
    }, {});

    useEffect(() => {
        const { protocol, hostname, pathname, href } = window.location;
        const { query } = url.parse(href, true);

        const path = url.format({
            protocol,
            hostname,
            pathname,
            query: {
                ...query,
                ...checkedFilters
            }
        });

        window.history.pushState({ path }, window.document.title, path);
    }, [checkedFilters]);

    const { HtmlContent } = useComponents();

    const dispatch = useDispatch();

    function changeServiceFilter(optionId: string) {
        if (checkedFilters.service && checkedFilters.service.includes(optionId)) {
            dispatch(removeMultiFilter('service' as TMultiValueFilterId, optionId));
        } else {
            dispatch(addMultiFilter('service' as TMultiValueFilterId, optionId));
        }
    }

    return (
        <div className={b({}, [className])}>
            <div className={b('region-switcher')}>
                <span className={b('region-switcher-label')}>
                    {i18n('region-choose')}
                </span>
                <RegionSwitcher tld={tld} regions={regions} hasFlags />
            </div>
            <div className={b('title')}>
                <FormattedContent
                    template={filtersTemplate}
                    nodes={filterNodes}
                    />
                <div className={b('list-service')}>
                    {services.map(service => {
                        const classNameService = checkedFilters.service && checkedFilters.service.includes(service.id) ? b('service-select') : b('service-not-select');

                        return (
                            <div className={`${b('service-item')} ${classNameService}`} onClick={() => { changeServiceFilter(service.id); }} key={service.id}>
                                {service.name}
                            </div>
                        );
                    })}
                </div>
            </div>
            {description && (
                <HtmlContent
                    className={b('description')}
                    theme="light"
                    content={description}
                    />
            )}
            {hasBudgetFilter && <BudgetFilter />}
        </div>
    );
}

interface IFilterGroupProps {
    id: TFilterId;
    options: IFilterOption[];
    allowMultiple?: boolean;
    relatedFilter?: IAgenciesFilter;
}

function FiltersGroup({ id: filterId, options, allowMultiple = true, relatedFilter }: IFilterGroupProps) {
    const { Popup } = useLegoComponents();
    const filters = useAgenciesFilters();
    const dispatch = useDispatch();
    const checked = ([] as string[]).concat(filters[filterId]!);

    const popupToggler = useToggle();

    const isAllChecked = checked.includes('all');
    const unchecked = options
        .map(option => option.id)
        .filter(id => !checked.includes(id));

    const canAddFilters = allowMultiple && !isAllChecked && unchecked.length > 0;

    const buttonRef = useRef<HTMLDivElement>(null);

    function createRemoveHandler(optionId: string) {
        return () => {
            dispatch(removeMultiFilter(filterId as TMultiValueFilterId, optionId));
        };
    }

    function createReplaceHandler(currentIndex: number) {
        return (optionId: string) => {
            reachFilterGoal(filterId, optionId);

            if (allowMultiple) {
                dispatch(replaceMultiFilter(filterId as TMultiValueFilterId, currentIndex, optionId));
            } else {
                dispatch(changeSingleFilter(filterId as TSingleValueFilterId, optionId, relatedFilter));
            }
        };
    }

    function createAddHandler(optionId: string) {
        return () => {
            reachFilterGoal(filterId, optionId);
            dispatch(addMultiFilter(filterId as TMultiValueFilterId, optionId));
        };
    }

    return (
        <div className={b('group')}>
            {checked.map((id, index) => {
                const delimiter = getDelimiter(index, checked.length);

                return (
                    <Fragment key={id}>
                        {delimiter && <span className={b('delimiter')} dangerouslySetInnerHTML={{ __html: delimiter }} />}
                        <Filter
                            preposition={index === 0}
                            filterId={filterId}
                            option={findOptionById(options, id)}
                            hasRemove={checked.length > 1}
                            onRemove={createRemoveHandler(id)}
                            onReplace={createReplaceHandler(index)}
                            unchecked={unchecked}
                            options={options}
                            />
                    </Fragment>
                );
            })}
            {canAddFilters && (
                <Fragment>
                    <div
                        ref={buttonRef}
                        className={b('add-button')}
                        onClick={popupToggler.switchOn}
                        />
                    <Popup
                        target="anchor"
                        theme="normal"
                        anchor={buttonRef}
                        visible={popupToggler.value}
                        onOutsideClick={popupToggler.switchOff}
                        >
                        <div className={b('popup-content')}>
                            {unchecked.map(id => (
                                <div
                                    className={b('popup-item')}
                                    key={id}
                                    onClick={createAddHandler(id)}
                                    >
                                    {findOptionById(options, id).name}
                                </div>
                            ))}
                        </div>
                    </Popup>
                </Fragment>
            )}
        </div>
    );
}

function getDelimiter(index: number, sequenceLength: number) {
    if (index === 0) {
        return '';
    }

    if (index === sequenceLength - 1) {
        return i18n('filters-conjunction');
    }

    return i18n('filters-delimiter');
}

interface IFilterProps {
    filterId: string;
    hasRemove?: boolean;
    preposition: boolean;
    unchecked: string[];
    option: IFilterOption;
    options: IFilterOption[];
    onRemove?(): void;
    onReplace(id: string): void;
}

function Filter({
    filterId,
    hasRemove = false,
    onRemove,
    onReplace,
    preposition,
    unchecked,
    option,
    options
}: IFilterProps) {
    const { Popup } = useLegoComponents();

    const filterRef = useRef(null);
    const popupToggler = useToggle();

    const text = (preposition && option.preposition) ?
        `${option.preposition}&nbsp;${option.namePrepositional}` :
        option.namePrepositional!;

    return (
        <Fragment>
            <span
                ref={filterRef}
                className={b('filter', { type: filterId })}
                onClick={popupToggler.switchOn}
                dangerouslySetInnerHTML={{ __html: text }}
                />
            <Popup
                target="anchor"
                theme="normal"
                anchor={filterRef}
                visible={popupToggler.value}
                onOutsideClick={popupToggler.switchOff}
                >
                <div className={b('popup-content')}>
                    {hasRemove && (
                        <div
                            className={b('popup-item', { type: 'remove' })}
                            onClick={onRemove}
                            >
                            {option.name}
                        </div>
                    )}
                    {unchecked.map(id => (
                        <div
                            className={b('popup-item')}
                            key={id}
                            onClick={() => onReplace(id)}
                            >
                            {findOptionById(options, id).name}
                        </div>
                    ))}
                </div>
            </Popup>
        </Fragment>
    );
}

function BudgetFilter() {
    const { Textinput, Button, Select } = useLegoComponents();

    const { budget, budgetOrder } = useAgenciesFilters();
    const dispatch = useDispatch();

    const [budgetFilter, setBudgetFilter] = useState<TBudget>(budget);

    /* Сбрасываем локальный стейт при сбросе фильтров */
    useEffect(() => {
        if (budget === undefined) {
            setBudgetFilter(budget);
        }
    }, [budget]);

    function updateState() {
        dispatch(setBudget(budgetFilter));

        if (!budgetOrder) {
            dispatch(setBudgetOrder('asc'));
        }
    }

    const budgetNum = Number(budgetFilter);
    const isCorrectBudget = Number.isInteger(budgetNum) && budgetNum >= 0;

    function onBudgetChange(event: ChangeEvent<HTMLInputElement>) {
        const value = event.target.value === '' ? undefined : event.target.value;

        setBudgetFilter(value);
    }

    function onBudgetOrderChange(event: ChangeEvent<HTMLSelectElement>) {
        dispatch(setBudgetOrder(event.target.value));
    }

    const budgetOrderValue = budgetOrder || 'asc';

    return (
        <div className={b('filter', { type: 'budget' })}>
            <div className={b('budget-filter')}>
                <span
                    className={b('budgets')}
                    dangerouslySetInnerHTML={{ __html: `${i18n('budgets')}&nbsp;` }}
                    />
                <div className={b('budget-input')}>
                    <div className={b('input-wrapper')}>
                        <Textinput
                            baseline
                            // @ts-ignore
                            min="0"
                            onChange={onBudgetChange}
                            pattern="[0-9]*"
                            pin="round-round"
                            size="s"
                            theme="normal"
                            type="number"
                            value={budgetFilter || ''}
                            />
                    </div>
                    <span
                        className={b('input-currency')}
                        dangerouslySetInnerHTML={{ __html: `&nbsp;${i18n('currency')}&nbsp;` }}
                        />
                </div>
                <Button
                    className={b('show-button')}
                    disabled={!isCorrectBudget}
                    onClick={updateState}
                    size="s"
                    theme="action"
                    >
                    {i18n('show-text')}
                </Button>
            </div>
            {budget && (
                <Select
                    className={b('budget-order')}
                    size="m"
                    theme="normal"
                    onChange={onBudgetOrderChange}
                    value={budgetOrderValue}
                    options={[
                        { value: 'asc', content: i18n('budget-asc-order') },
                        { value: 'desc', content: i18n('budget-desc-order') }
                    ]}
                    />
            )}
        </div>
    );
}

export * from './types';
export default AgenciesFilters;
