import React, {
    useEffect,
    useRef
} from 'react';
import { useDispatch } from 'react-redux';

import {
    useAgencies,
    useAgenciesFilters,
    useComponents,
    useConfig,
    useLegoComponents
} from 'client/common/hooks';

import cn from 'utils/cn';
import { bindKeyset } from 'utils/i18n';
import { isElementScrolledToEnd } from 'utils/scroll';
import { throttle } from 'utils/throttle';

import {
    failLoading,
    finishLoading,
    resetFilters,
    startLoading,
    startLoadingFromScratch
} from 'store/agencies/actions';

import { IProps } from './types';

import './index.css';

const b = cn('agencies-list');
const i18n = bindKeyset('agencies');

function AgenciesList(props: IProps) {
    const { Button, Spin } = useLegoComponents();
    const { AgencyInfo } = useComponents();

    const listRef = useRef<HTMLDivElement>(null);

    const abortController = useRef<AbortController>();

    const { secretKey } = useConfig();
    const initialRender = useRef(true);

    const {
        visible,
        defaultFilters,
        filterUrl,
        group,
        resetTextKey
    } = props;
    const { lists, isError } = useAgencies();
    const { agencies, hasMore, isLoading, seed } = lists[group];

    const filters = useAgenciesFilters();
    const reduxDispatch = useDispatch();
    const handleResetFilters = () => {
        reduxDispatch(resetFilters(defaultFilters));
    };

    /* Загружаем агенства при изменении флага */
    useEffect(() => {
        if (!isLoading || !hasMore) {
            return;
        }

        if (abortController.current) {
            abortController.current.abort();
        }

        abortController.current = new AbortController();

        fetch(filterUrl, {
            method: 'POST',
            signal: abortController.current.signal,
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-Token': secretKey
            },
            body: JSON.stringify({
                filters,
                seed,
                offset: agencies.length,
                group
            })
        })
            .then(response => response.json())
            .then(response => {
                abortController.current = undefined;

                reduxDispatch(finishLoading({
                    agencies: response.agencies,
                    hasMore: response.hasMore,
                    group
                }));
            })
            .catch(error => {
                console.error(error);

                reduxDispatch(failLoading(group));
            });
    }, [
        agencies,
        filters,
        filterUrl,
        hasMore,
        isLoading,
        secretKey,
        seed,
        group,
        reduxDispatch
    ]);

    /* Инициируем загрузку при скролле */
    useEffect(() => {
        function handleScroll() {
            if (!listRef.current || !visible) {
                return;
            }

            if (isElementScrolledToEnd(listRef.current, 0.7) && hasMore && !isLoading && !isError) {
                reduxDispatch(startLoading(group));
            }
        }

        const throttledHandleScroll = throttle(handleScroll, 300);

        window.addEventListener('scroll', throttledHandleScroll);

        return () => {
            window.removeEventListener('scroll', throttledHandleScroll);
        };
    }, [hasMore, visible, group, isLoading, reduxDispatch, isError]);

    /* Инициируем загрузку при изменении фильтров */
    useEffect(() => {
        if (initialRender.current) {
            initialRender.current = false;
        } else {
            reduxDispatch(startLoadingFromScratch(group));
        }
    }, [filters, initialRender, group, reduxDispatch]);

    return (
        <div className={b({ group })} ref={listRef}>
            {agencies.map(agency => {
                return (
                    <div key={agency.slug} className={b('item-wrapper')}>
                        <div className={b('item')}>
                            <AgencyInfo agency={agency} />
                        </div>
                    </div>
                );
            })}
            {isLoading && (
                <div className={b('spinner')}>
                    <Spin
                        progress
                        size="l"
                        view="default"
                        />
                </div>
            )}
            {!isLoading && agencies.length === 0 && resetFilters && resetTextKey && !isError && (
                <div className={b('filters-reset')}>
                    <h3 className={b('reset-label')}>{i18n(resetTextKey)}</h3>
                    <Button
                        className={b('reset-button')}
                        size="m"
                        theme="normal"
                        type="link"
                        onClick={handleResetFilters}
                        >
                        {i18n('filters-reset')}
                    </Button>
                </div>
            )}
        </div>
    );
}

export default AgenciesList;
