import compact from 'lodash/compact';

import { IAgenciesFilters } from 'client/common/types';
import { TAgenciesFilters } from 'store/agencies/types';

import { bindKeyset } from 'utils/i18n';

import { IAgenciesFilterText, IAgenciesFilterTexts } from './AgenciesTitle.types';

const i18n = bindKeyset('agencies');

function buildFilterText(filters: IAgenciesFilterText[]): string {
    filters = compact(filters);
    if (!filters.every(filter => filter.namePrepositional)) {
        return '';
    }

    return filters
        .map((filterData, index) => {
            let delimiter: string;

            if (index === 0) {
                delimiter = filterData.preposition || '';
            } else if (index === filters.length - 1) {
                delimiter = i18n('filters-conjunction');
            } else {
                delimiter = i18n('filters-delimiter');
            }

            return `${delimiter} ${filterData.namePrepositional}`;
        })
        .join('');
}

function getTokenName(string: string, token: RegExp) {
    if (!isToken(string, token)) {
        return null;
    }
    const match = string.match(token);

    return match && match[1];
}

function isToken(string: string, token: RegExp) {
    return token.test(string);
}

export function buildTitle(filters: TAgenciesFilters, rawFilters: IAgenciesFilters, template: string) {
    const certSelection = filters.certificate.map(cert => rawFilters.certificates.find(rawCert => rawCert.code === cert));

    const countriesSelection = rawFilters.countries.length > 1 ?
        [rawFilters.countries.find(country => String(country.geoId) === filters.country)]
        : [];

    const citiesSelection = filters.city.includes('all') ?
        [{
            namePrepositional: i18n('filter-all-cities-name-prepositional'),
            preposition: i18n('filter-all-cities-preposition')
        }]
        : filters.city.map(city => rawFilters.cities.find(rawCity => String(rawCity.geoId) === city));

    const [certText, countriesText, citiesText] =
        [certSelection, countriesSelection, citiesSelection].map(s => buildFilterText(s as IAgenciesFilterText[]));

    const nodes: IAgenciesFilterTexts = {
        country: countriesText,
        city: citiesText,
        certificate: certText
    };

    const TOKEN_REG_EXP = /({{\w+}})/g;
    const TOKEN_NAME_REG_EXP = /{{(\w+)}}/;

    return template.split(TOKEN_REG_EXP)
        .map(part => {
            const tokenName = getTokenName(part, TOKEN_NAME_REG_EXP);

            return tokenName ? nodes[tokenName as keyof(IAgenciesFilterTexts)] : part;
        })
        .join('');
}
