import React, {
    ChangeEvent,
    Dispatch,
    Fragment,
    MouseEvent,
    ReducerAction,
    useCallback,
    useReducer,
    useState
} from 'react';

import EmailSample from 'client/common/components/email-sample';
import { useAgencyOffices, useConfig, useLegoComponents } from 'client/common/hooks';

import cn from 'utils/cn';
import i18n from 'utils/i18n';
import { reachGoal } from 'utils/metrika';

import { getInitialState, reducer } from './reducer';
import { ActionType } from './types';

import './index.css';

const b = cn('agency-form');

type TLegoComponents = ReturnType<typeof useLegoComponents>;

interface IAgencyFormRowProps {
    dispatch: Dispatch<ReducerAction<typeof reducer>>;
    disabled: boolean;
    errorText: string;
    errorVisible: boolean;
    fieldName: string;
    InputComponent: TLegoComponents['Textinput'] | TLegoComponents['Textarea'];
    inputText: string;
    inputType?: string;
    labelText: string;
    required?: boolean;
    typeMod?: 'full';
}

function AgencyFormRow({
    dispatch,
    disabled,
    errorText,
    errorVisible,
    fieldName,
    InputComponent,
    inputText,
    inputType,
    labelText,
    required,
    typeMod
}: IAgencyFormRowProps) {
    const handleClick = useCallback((event: MouseEvent<HTMLSpanElement>) => dispatch({
        type: ActionType.SetFieldValue,
        payload: {
            fieldName,
            value: (event.target as HTMLSpanElement).textContent!
        }
    }), [dispatch, fieldName]);

    const handleChange = useCallback((event: ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
        dispatch({
            type: ActionType.SetFieldValue,
            payload: {
                fieldName,
                value: event.target.value
            }
        });

        if (errorVisible) {
            dispatch({
                type: ActionType.SetFieldError,
                payload: {
                    fieldName,
                    error: false
                }
            });
        }
    }, [dispatch, errorVisible, fieldName]);

    return (
        <div className={b('row', { type: inputType }, ['clearfix'])}>
            <div className={b('label', { type: typeMod })}>
                <div className={b('label-text', { required })}>
                    {labelText}
                </div>
            </div>
            <div className={b('control', { type: typeMod })}>
                <InputComponent
                    addonAfter={(
                        <Fragment>
                            {inputType === 'email' && <EmailSample onClick={handleClick} />}
                            <span className={b('error-text', { visible: errorVisible })}>
                                {errorText}
                            </span>
                        </Fragment>
                    )}
                    onChange={handleChange}
                    name={fieldName}
                    type={inputType}
                    value={inputText}
                    size="m"
                    theme="normal"
                    // @ts-ignore
                    required={required}
                    disabled={disabled}
                    hasClear
                    />
            </div>
        </div>
    );
}

interface IAgencyFormProps {
    agencySlug: string;
}

function AgencyForm({ agencySlug }: IAgencyFormProps) {
    const { secretKey, contactFormPath } = useConfig();

    const { Spin, Button, Textinput, Textarea } = useLegoComponents();
    const requiredError = i18n({ keyset: 'partners', key: 'form.error.required' });
    const emailError = i18n({ keyset: 'partners', key: 'form.error.email' });

    const [state, dispatch] = useReducer(reducer, getInitialState());

    const [sending, setSending] = useState(false);
    const [sent, setSent] = useState(false);
    const [errorMessage, setErrorMessage] = useState('');

    const officesState = useAgencyOffices();

    function validateForm() {
        let errorsCount = 0;

        for (const fieldName of Object.keys(state)) {
            const field = state[fieldName];

            if (!field.value) {
                errorsCount += 1;

                dispatch({
                    type: ActionType.SetFieldError,
                    payload: {
                        fieldName,
                        error: true
                    }
                });
            }

            const isEmail = field.isEmail && field.value;
            const isEmailError = isEmail && !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(field.value);

            if (isEmailError) {
                errorsCount += 1;

                dispatch({
                    type: ActionType.SetFieldError,
                    payload: {
                        fieldName,
                        error: true
                    }
                });
            }
        }

        return errorsCount;
    }

    function handleSubmit(event: MouseEvent) {
        event.preventDefault();

        reachGoal('send_request_to_agency');

        const errorsCount = validateForm();

        if (errorsCount > 0) {
            return;
        }

        const data = new FormData();

        data.append('formName', 'agency-form');
        data.append('agency', JSON.stringify({
            slug: agencySlug,
            officeId: officesState.selectedId
        }));

        const form = Object
            .keys(state)
            .reduce((acc, fieldName) => {
                const fieldData = Object.assign({ name: fieldName }, {
                    label: state[fieldName].label,
                    isEmail: state[fieldName].isEmail,
                    isSelect: state[fieldName].isSelect,
                    value: state[fieldName].value
                });

                // @ts-ignore
                return acc.concat(fieldData);
            }, []);

        data.append('form', JSON.stringify(form));

        data.append('type', 'translation');

        setSending(true);
        setErrorMessage('');

        fetch(contactFormPath, {
            method: 'POST',
            headers: {
                'x-csrf-token': secretKey
            },
            cache: 'no-cache',
            body: data
        })
            .then(response => {
                if (response.ok) {
                    return setSent(true);
                }

                return response.json();
            })
            .then(json => {
                const message = i18n({ keyset: 'forms', key: json.internalCode }) || json.message;

                return Promise.reject(new Error(message));
            })
            .catch(({ message }: Error) => {
                setErrorMessage(message);
            })
            .finally(() => setSending(false));
    }

    return (
        <form className={b()}>
            <div className={b('data', { hidden: sent })}>
                <AgencyFormRow
                    dispatch={dispatch}
                    disabled={sending}
                    errorText={requiredError}
                    errorVisible={state.name.error}
                    fieldName="name"
                    InputComponent={Textinput}
                    inputText={state.name.value}
                    inputType="text"
                    labelText={i18n({ keyset: 'agency', key: 'form.name' })}
                    required
                    />
                <AgencyFormRow
                    dispatch={dispatch}
                    disabled={sending}
                    errorText={emailError}
                    errorVisible={state.email.error}
                    fieldName="email"
                    InputComponent={Textinput}
                    inputText={state.email.value}
                    inputType="email"
                    labelText={i18n({ keyset: 'agency', key: 'form.email' })}
                    required
                    />
                <AgencyFormRow
                    dispatch={dispatch}
                    disabled={sending}
                    errorText={requiredError}
                    errorVisible={state.comment.error}
                    fieldName="comment"
                    InputComponent={Textarea}
                    inputText={state.comment.value}
                    labelText={i18n({ keyset: 'agency', key: 'form.message' })}
                    typeMod="full"
                    required
                    />
                <div className={b('footer')}>
                    <div className={b('error')}>
                        <div className={b('error-message', { visible: Boolean(errorMessage) })}>
                            {errorMessage}
                        </div>
                    </div>
                    {sending && (
                        <Spin
                            progress
                            view="default"
                            size="m"
                            position="center"
                            />
                    )}
                    <div className={b('button')}>
                        <Button theme="action" size="m" type="submit" onClick={handleSubmit}>
                            {i18n({ keyset: 'agency', key: 'form.button' })}
                        </Button>
                    </div>
                </div>
            </div>
            <div className={b('message', { hidden: !sent })}>
                <div className={b('message-title')}>
                    {i18n({ keyset: 'forms', key: 'success.agency.title' })}
                </div>
                <p>
                    {i18n({ keyset: 'forms', key: 'success.agency.text' })}
                </p>
            </div>
        </form>
    );

}

export default AgencyForm;
