import React from 'react';
import { Link } from 'react-router-dom';

import { SiteIcon } from 'client/common/components/icons';
import { useAgenciesFilters, useComponents, useConfig } from 'client/common/hooks';

import { CertificateType, IAgency, IEnumValue } from 'client/common/types';

import { getLogo } from 'utils/agency';
import cn from 'utils/cn';
import { getUniqueValues } from 'utils/helper';
import { bindKeyset } from 'utils/i18n';
import { reachGoal } from 'utils/metrika';
import { formatInteger } from 'utils/number';

import './index.css';

const b = cn('agency-info');
const i18nAgency = bindKeyset('agency');
const i18nAgencies = bindKeyset('agencies');

interface IAgencyInfoProps {
    agency: IAgency;
}
interface IAgencyCertificateList {
    certificates: Array<IEnumValue<CertificateType>>;
    directCompanyType: string | null;
}

function CertificateList({ certificates }: IAgencyCertificateList) {
    const filter = useAgenciesFilters();

    const { AgencyEstimate } = useComponents();

    certificates = certificates.sort((current, next) => {
        return filter.certificate.indexOf(next.code) - filter.certificate.indexOf(current.code);
    });

    return (
        <div className={b('certificates')}>
            <span className={b('certificates-label')}>
                {`${i18nAgency('certificates')}: `}
            </span>
            {
                certificates.map((certificate, index) => {

                    return (
                        <div className={b('certificates-item')} key={certificate.code}>
                            <span className={b('certificates-name')}> {certificate.name}</span>
                            {
                                certificate.estimate &&
                                    <AgencyEstimate estimate={certificate.estimate} />
                            }
                            {
                                index === 1 && certificates.length > 2 &&
                                <span>
                                    {
                                        i18nAgency('more') ?
                                            `${i18nAgency('more')} ${certificates.length - 2}` :
                                            '...'
                                    }
                                </span>
                            }
                        </div>
                    );
                })
            }
        </div>
    );
}

function AgencyInfo({ agency }: IAgencyInfoProps) {
    const {
        directBudget,
        directCompanyType,
        logo,
        name: title,
        offices,
        site,
        slug,
        productsInfo
    } = agency;

    let { certificates } = agency;

    const { agenciesConfig } = useConfig();
    const filters = useAgenciesFilters();

    const officesChosenByCity = filters.city.includes('all') ?
        offices :
        offices.filter(office => filters.city.includes(String(office.city.geoId)));

    const cityNames = officesChosenByCity.map(office => office.city.name);
    const uniqueCityNames = getUniqueValues(cityNames).join(', ');

    certificates = certificates.map(certificate => {
        if (certificate.code === 'direct' && directCompanyType && directCompanyType === 'bidder') {
            certificate.name = i18nAgency(`type.${directCompanyType}`);
        }

        if (productsInfo && productsInfo.length) {
            const productsInfoCert = productsInfo?.find(productsInfoItem => productsInfoItem.product_slug === certificate.code);

            certificate.estimate = productsInfoCert?.metrics?.find(metric => metric.slug === productsInfoCert.total_slug)?.value;

            return certificate;
        }

        return certificate;
    });

    function onCardClick() {
        reachGoal('agency-to-page');
        reachGoal('PODR');
    }

    function onSiteClick() {
        reachGoal('agency-click-to-website');
    }

    const formattedDirectBudget = formatInteger(directBudget);
    const budgetText = `${i18nAgencies('budget-from')}&nbsp;${formattedDirectBudget}&nbsp;${i18nAgencies('currency')}`;

    return (
        <article className={b()}>
            <div className={b('main')}>
                <div className={b('logo')}>
                    <img className={b('image')} src={getLogo(logo, agenciesConfig, 'm')} />
                </div>
                <div className={b('info')}>
                    <h3 className={b('title')} dangerouslySetInnerHTML={{ __html: title }} />
                    <div className={b('cities')}>{uniqueCityNames}</div>
                    {certificates.length > 0 && <CertificateList certificates={certificates} directCompanyType={directCompanyType} />}
                    <Link
                        className={b('card-link')}
                        onClick={onCardClick}
                        to={`/contact/agencies/${slug}`}
                        />
                    <Link
                        className={b('pseudo-card-link')}
                        onClick={onCardClick}
                        to={`/contact/agencies/${slug}`}
                        >
                        {i18nAgency('card-link')}
                    </Link>
                    <a
                        className={b('site-link')}
                        href={`http://${site}`}
                        target="_blank"
                        rel="noopener noreferrer"
                        onClick={onSiteClick}
                        >
                        <SiteIcon />
                    </a>
                </div>
            </div>
            {filters.budget && (
                <div
                    className={b('budget-text')}
                    dangerouslySetInnerHTML={{ __html: budgetText }}
                    />
            )}
        </article>
    );
}

export default AgencyInfo;
