import React, { ChangeEvent, Fragment, ReactNode } from 'react';
import { useDispatch } from 'react-redux';

import AgencyForm from 'client/common/components/agency-form';

import { useAgencyForm, useAgencyShowProductsCard, useComponents, useConfig, useLegoComponents } from 'client/common/hooks';

import { CertificateType, Country, IAgency, IAgencyMetrics, IAgencyProductsInfo } from 'client/common/types';
import { LoadingState } from 'client/common/types/common';

import {
    cancelEditing,
    changeFormData,
    hideFormPreview,
    sendingFail,
    sendingStart,
    sendingSuccess,
    showFormPreview,
    showOrHideProductsCard,
    startEditing
} from 'store/agency/actions';
import { AgencyFormMode, IAgencyState } from 'store/agency/types';

import { getFilledEmail, getLogo, hasCertificate } from 'utils/agency';
import cn from 'utils/cn';
import { bindKeyset } from 'utils/i18n';
import { formatInteger } from 'utils/number';

import './index.css';

const b = cn('agency');
const i18n = bindKeyset('agency');

interface IAgencyContainerProps {
    agency: IAgency;
    agenciesPageUrl?: string;
    agencyUrl: string;
    children: ReactNode;
    metrics: {

    };
}

function Agency({ agency, agenciesPageUrl, children }: IAgencyContainerProps) {
    const { certificates, isPartner } = agency;
    const isActive = certificates.length > 0 || isPartner;

    // ADVDEV-1542: Получать с бэка информацию о том, можно ли редактировать агентство
    const canEdit = false;
    const { HtmlContent } = useComponents();

    return (
        <section className={b()}>
            {canEdit && <AgencyFormStatus agency={agency} />}
            {isActive ?
                children :
                (
                    <div className={b('non-active')}>
                        <HtmlContent
                            className={b('non-active-text')}
                            theme="light"
                            content={i18n('non-active-agency')}
                            />
                        <HtmlContent
                            className={b('agencies-with-certificates-text')}
                            theme="light"
                            content={i18n('agencies-with-certificates', { agenciesPageUrl })}
                            />
                    </div>
                )
            }
        </section>
    );
}

interface IAgencyFormStatus {
    agency: IAgency;
}

function AgencyFormStatus({ agency }: IAgencyFormStatus) {
    const form = useAgencyForm();
    const dispatch = useDispatch();

    function onEditButtonClick() {
        dispatch(startEditing(agency));
    }

    function onBackToEditButtonClick() {
        dispatch(hideFormPreview());
    }

    return (
        <div className={b('form-status')}>
            {form.mode === AgencyFormMode.hidden && (
                form.sendingState === LoadingState.success ?
                    i18n('on-moderation') :
                    <div className={b('edit-button')} onClick={onEditButtonClick}>
                        {i18n('edit')}
                    </div>
            )}
            {form.mode === AgencyFormMode.preview &&
                <div className={b('back-to-edit-button')} onClick={onBackToEditButtonClick}>
                    {i18n('back-to-edit')}
                </div>
            }
            {form.mode === AgencyFormMode.editing &&
                i18n('editing-mode')
            }
        </div>
    );
}

interface IAgencyPlateProps {
    directCompanyType?: string | null;
    offices: IAgency['offices'];
    showPartner: boolean;
}

interface IAgencyCertText {
    main: string;
    footer?: string;
}

function getLocaleKeyByCompanyType(directCompanyType: string) {
    switch (directCompanyType) {
        case 'bidder':
            return 'bidder';
        case 'qualified-partner':
            return 'qualified-yandex-partner';
        case 'partner':
            return 'yandex-direct-partner';
        case 'leading-partner':
            return 'yandex-leading-partner';
        default:
            return directCompanyType;
    }
}

function getAgetncyCertText(directCompanyType: string | null | undefined, showPartner: boolean, countryGeoId: number): IAgencyCertText {
    const localizedType = directCompanyType && i18n(getLocaleKeyByCompanyType(directCompanyType));

    if (localizedType) {
        return {
            main: localizedType
        };
    }

    if (showPartner) {
        return {
            main: i18n('yandex-partner')
        };
    }

    return [Country.Belarus, Country.Kazakhstan, Country.Russia].includes(countryGeoId) ? {
        main: i18n('certified-yandex-agency'),
        footer: i18n('cert-by-yandex')
    } : {
        main: i18n('qualified-yandex-partner')
    };
}

function AgencyPlate({ directCompanyType, offices, showPartner }: IAgencyPlateProps) {
    const { tld } = useConfig();

    const mainOffice = offices.find(office => office.isMain) || offices[0];
    const countryGeoId = mainOffice.city.country!.geoId;

    const certText = getAgetncyCertText(directCompanyType, showPartner, countryGeoId);
    const showFooter = Boolean(certText.footer);

    const letterMods = { tld: tld.replace('.', '-') };
    const certTextLines = certText.main.split('<br>');

    return (
        <div className={b('plate')}>
            <div className={b('plate-letter', letterMods)} />
            <div className={b('plate-text')}>
                { certTextLines.map(line =>
                    (
                        <div key={line} className={b('plate-main', { 'with-footer': showFooter })}>
                            { line }
                        </div>
                    )
                ) }
                { showFooter &&
                    <div className={b('plate-footer')}>
                        { certText.footer }
                    </div> }
            </div>
        </div>
    );
}

interface IAgencyLeftColumnProps {
    certificates: IAgency['certificates'];
    directCompanyType: string | null;
    isPartner: boolean;
    logo: string;
    offices: IAgency['offices'];
    title: string;
    productsInfo: IAgencyProductsInfo[];
    metrics: IAgencyMetrics;
}

function AgencyLeftColumn({
    certificates,
    directCompanyType,
    isPartner,
    logo,
    offices,
    title,
    metrics,
    productsInfo
}: IAgencyLeftColumnProps) {
    const { agenciesConfig } = useConfig();
    const showPartner = certificates.length === 0 && isPartner;
    const hasDirectCertificate = hasCertificate(certificates, CertificateType.Direct);

    return (
        <div className={b('left-column')}>
            <div className={b('logo')}>
                <img src={getLogo(logo, agenciesConfig)} alt={title} />
            </div>
            <AgencyPlate
                directCompanyType={hasDirectCertificate ? directCompanyType : undefined}
                offices={offices}
                showPartner={showPartner}
                />
            {certificates.length > 0 && <AgencyCertificates certificates={certificates} metrics={metrics} productsInfo={productsInfo} agencyName={title} />}
        </div>
    );
}

interface IAgencyCertificatesProps {
    certificates: IAgency['certificates'];
    productsInfo: IAgencyProductsInfo[];
    metrics: IAgencyMetrics;
    agencyName: string;
}

function AgencyCertificates({ certificates, productsInfo, metrics, agencyName }: IAgencyCertificatesProps) {
    const { AgencyEstimate, ProductsInfoCard } = useComponents();
    const showProductsCard = useAgencyShowProductsCard() || {
        market: false
    };
    const dispatch = useDispatch();

    function setShowOrHideProductsCard(certificate: 'market'| 'direct' | 'metrika' | 'toloka' |'dialogs', show: boolean) {
        dispatch(showOrHideProductsCard(certificate, show));
    }

    return (
        <div className={b('certificates')}>
            <div className={b('spec-text')}>
                { i18n('specialization') }
            </div>
            {
                certificates.map(({ code }) => {
                    const productsInfoAgency = productsInfo?.find(productsInfoItem => productsInfoItem.product_slug === code);
                    const metricAgency = productsInfoAgency?.metrics;
                    const estimate = metricAgency?.find(metric => metric.slug === productsInfoAgency?.total_slug)?.value;

                    return (
                        <div key={code} className={b('certificate')}>
                            <div className={b('certificate-wrapper')}>
                                { i18n(`type.${code}`) }
                                { estimate && <AgencyEstimate estimate={estimate} />}
                                {
                                    productsInfoAgency && metricAgency && metrics && estimate &&
                                    <div className={`${b('products-info-card-wrapper')} ${!showProductsCard[code] && b('products-info-card-wrapper-display-none')}`}>
                                        { <ProductsInfoCard
                                            metricAgency={metricAgency}
                                            metrics={metrics}
                                            agencyName={agencyName}
                                            totalMetric={estimate}
                                            totalMetricSlug={productsInfoAgency.total_slug}
                                            />}
                                    </div>
                                }
                                {
                                    metricAgency && metrics &&
                                    <button
                                        className={b('information-button')}
                                        onClick={
                                            () => {
                                                setShowOrHideProductsCard(code, !showProductsCard[code]);
                                            }
                                        }
                                        >
                                        {'i'}
                                    </button>
                                }
                            </div>
                        </div>
                    );
                })
            }
        </div>
    );
}

interface IAgencyWrapperProps {
    agency: IAgency;
    agencyUrl: string;
    children?: ReactNode;
    onButtonClick(): void;
}

function AgencyWrapper({
    agency,
    agencyUrl,
    children,
    onButtonClick
}: IAgencyWrapperProps) {
    const form = useAgencyForm();

    const isFormVisible = form.mode === AgencyFormMode.editing;

    const { slug } = agency;

    return (
        <div className={b('wrapper')}>
            <AgencyMainSection agency={agency} onButtonClick={onButtonClick}>
                {children}
            </AgencyMainSection>
            <AgencyOfficesSection agency={agency} />
            <AgencyMaterialsSection agency={agency} />
            {isFormVisible && <FormControls slug={slug} agencyUrl={agencyUrl} />}
            {form.sendingState === LoadingState.fail &&
                <div className={b('sending-fail')}>{i18n('sending-fail')}</div>
            }
        </div>
    );
}

interface IAgencyMainSectionProps {
    agency: IAgency;
    children?: ReactNode;
    onButtonClick(): void;
}

function AgencyMainSection({
    agency,
    children,
    onButtonClick
}: IAgencyMainSectionProps) {
    const { ExpandableText, UnderscoredInput } = useComponents();
    const { Button, Textarea } = useLegoComponents();
    const { agenciesConfig: { agency: agencyConfig }, tld } = useConfig();

    const dispatch = useDispatch();
    const form = useAgencyForm();

    const isFormVisible = form.mode === AgencyFormMode.editing;

    const {
        name,
        description,
        directBudget,
        offices
    } = getAgencyData(agency, form);

    const hasEmail = Boolean(getFilledEmail(offices));

    function isBudgetVisible() {
        // FIXME: Унести настройку в бункер или конфиг
        const isBudgetSupported = ['ru'].includes(tld);

        const { certificates } = agency;
        const hasDirectCertificate = hasCertificate(certificates, CertificateType.Direct);

        return isBudgetSupported && hasDirectCertificate && (directBudget !== null || isFormVisible);
    }

    function onNameChange(event: ChangeEvent<HTMLInputElement>) {
        dispatch(changeFormData({ name: event.target.value }));
    }

    function onDirectBudgetChange(event: ChangeEvent<HTMLInputElement>) {
        const { value } = event.target;
        const directBudgetValue = value ? Number(value) : null;

        if (directBudgetValue !== null && Number.isNaN(directBudgetValue)) {
            return;
        }

        dispatch(changeFormData({ directBudget: directBudgetValue }));
    }

    function onDescriptionChange(event: ChangeEvent<HTMLTextAreaElement>) {
        const descriptionValue = event.target.value;

        dispatch(changeFormData({ description: descriptionValue }));
    }

    return (
        <section className={b('section')}>
            <header className={b('header')}>
                {isFormVisible ?
                    <UnderscoredInput
                        size="m"
                        value={name}
                        placeholder={i18n('name-placeholder')}
                        onChange={onNameChange}
                        /> :
                    <h1
                        className={b('title')}
                        dangerouslySetInnerHTML={{ __html: name }}
                        />
                }
            </header>
            {isBudgetVisible() && (
                <div className={b('budget')}>
                    {isFormVisible ?
                        <UnderscoredInput
                            size="s"
                            value={budgetToString(directBudget)}
                            placeholder={i18n('budget-placeholder')}
                            onChange={onDirectBudgetChange}
                            /> :
                        <Fragment>
                            <div className={b('currency')} />
                            <span className={b('budget-value')}>
                                {i18n('budget', { budget: formatInteger(directBudget) })}
                            </span>
                        </Fragment>
                    }
                </div>
            )}
            {children}
            {isFormVisible ?
                <div className={b('description-form')}>
                    <Textarea
                        theme="normal"
                        size="m"
                        value={description}
                        onChange={onDescriptionChange}
                        addonAfter={<DescriptionCounters description={description} />}
                        />
                </div> :
                (description &&
                    <ExpandableText
                        className={b('description')}
                        text={description}
                        shortenedTextMaxLength={agencyConfig.description.maxPreviewLength}
                        />
                )
            }
            {!isFormVisible && hasEmail &&
                <div className={b('contact-button')}>
                    <Button theme="normal" size="m" onClick={onButtonClick}>
                        {i18n('feedback')}
                    </Button>
                </div>
            }
        </section>
    );
}

function getAgencyData(agency: IAgency, form: IAgencyState['form']) {
    const isEditing = form.mode !== AgencyFormMode.hidden;

    return (form.data && isEditing) ? form.data : agency;
}

interface IAgencyOfficesSectionProps {
    agency: IAgency;
}

function AgencyOfficesSection({ agency }: IAgencyOfficesSectionProps) {
    const { Offices } = useComponents();
    const form = useAgencyForm();

    const isFormVisible = form.mode === AgencyFormMode.editing;
    const { site, offices } = getAgencyData(agency, form);

    if (offices.length === 0 && !isFormVisible) {
        return null;
    }

    return (
        <section className={b('section')}>
            <Offices offices={offices} site={site} />
        </section>
    );
}

interface IAgencyMaterialsSectionProps {
    agency: IAgency;
}

function AgencyMaterialsSection({ agency }: IAgencyMaterialsSectionProps) {
    const { AgencyMaterials } = useComponents();

    const { materials } = agency;

    if (materials.length === 0) {
        return null;
    }

    return (
        <section className={b('section')}>
            <header className={b('subtitle')}>{i18n('materials')}</header>
            <AgencyMaterials className={b('materials')} materials={materials} />
        </section>
    );
}

function budgetToString(budget: IAgency['directBudget']) {
    return budget === null ? '' : String(budget);
}

interface IDescriptionCountersProps {
    description: string;
}

function DescriptionCounters({ description }: IDescriptionCountersProps) {
    const {
        agenciesConfig: {
            agency: {
                description: { maxPreviewLength, maxLength }
            }
        }
    } = useConfig();

    return (
        <div className={b('description-counters')}>
            <div className={b('preview-description-counter')}>
                {i18n('preview-description-counter', {
                    count: Math.min(maxPreviewLength, description.length),
                    maxCount: maxPreviewLength
                })}
            </div>
            <div className={b('full-description-counter')}>
                {i18n('full-description-counter', {
                    count: description.length,
                    maxCount: maxLength
                })}
            </div>
        </div>
    );
}

interface IFormControlsProps {
    slug: string;
    agencyUrl: string;
}

function FormControls({ slug, agencyUrl }: IFormControlsProps) {
    const { Button } = useLegoComponents();
    const { secretKey } = useConfig();

    const dispatch = useDispatch();
    const form = useAgencyForm();

    if (!form.data) {
        return null;
    }

    const data = {
        ...form.data,
        offices: form.data.offices.map(office => {
            return {
                ...office,
                city: office.city.geoId
            };
        })
    };

    function sendFormData() {
        return fetch(agencyUrl, {
            method: 'PATCH',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-Token': secretKey
            },
            body: JSON.stringify({
                slug,
                agency: data
            })
        })
            .then(response => {
                if (response.ok) {
                    return dispatch(sendingSuccess());
                }

                dispatch(sendingFail());
            });
    }

    function onSendToModerationClick() {
        dispatch(sendingStart());
        sendFormData();
    }

    function onShowPreviewClick() {
        dispatch(showFormPreview());
    }

    function onCancelEditingClick() {
        dispatch(cancelEditing());
    }

    const isControlsDisabled = form.sendingState === LoadingState.inProgress;

    return (
        <div className={b('form-controls')}>
            <div className={b('primary-form-controls')}>
                <Button
                    theme="action"
                    size="m"
                    onClick={onSendToModerationClick}
                    disabled={isControlsDisabled}
                    >
                    {i18n('send-to-moderation')}
                </Button>
                <Button
                    theme="normal"
                    size="m"
                    onClick={onShowPreviewClick}
                    disabled={isControlsDisabled}
                    >
                    {i18n('show-preview')}
                </Button>
            </div>
            <div className={b('secondary-form-controls')}>
                <Button
                    theme="normal"
                    size="m"
                    onClick={onCancelEditingClick}
                    disabled={isControlsDisabled}
                    >
                    {i18n('cancel-editing')}
                </Button>
            </div>
        </div>
    );
}

interface IAgencyContentProps {
    children: ReactNode;
}

function AgencyContent({ children }: IAgencyContentProps) {
    return (
        <div className={b('content')}>
            {children}
        </div>
    );
}

interface IAgencyColProps {
    children: ReactNode;
}

function AgencyCol({ children }: IAgencyColProps) {
    return (
        <div className={b('col')}>
            {children}
        </div>
    );
}

interface IAgencyModalProps {
    agencySlug: string;
    children?: ReactNode;
    visible: boolean;
    handleClose(): void;
}

function AgencyModal({ agencySlug, handleClose, visible }: IAgencyModalProps) {
    const { ModalContent } = useComponents();
    const { Modal: LegoModal } = useLegoComponents();

    // Lego bug in Chrome: клик в браузерный саджест обрабатывается как нажатие Esc
    function onKeyDown(event: KeyboardEvent) {
        if (event.keyCode === 27) {
            handleClose();
        }
    }

    return (
        <div className={b('modal')}>
            <LegoModal
                visible={visible}
                onEscapeKeyDown={onKeyDown}
                onOutsideClick={handleClose}
                style={{ width: '100%', height: '100%' }}
                theme="normal"
                zIndexGroupLevel={20}
                >
                <ModalContent onCloseClick={handleClose} title={i18n('modal.title')}>
                    <AgencyForm agencySlug={agencySlug} />
                </ModalContent>
            </LegoModal>
        </div>
    );
}

export interface IAgencyProps {
    agency: IAgency;
    agencyUrl: string;
    agenciesPageUrl?: string;
    metrics: IAgencyMetrics;
}

Agency.Col = AgencyCol;
Agency.Content = AgencyContent;
Agency.LeftColumn = AgencyLeftColumn;
Agency.Modal = AgencyModal;
Agency.Wrapper = AgencyWrapper;

export default Agency;
