import 'client/common/polyfill';

import React, { useRef } from 'react';
import Helmet from 'react-helmet';
import { Provider } from 'react-redux';
import { Route, Switch } from 'react-router-dom';

import { cnTheme } from '@yandex-lego/components/Theme';
import { theme } from '@yandex-lego/components/Theme/presets/default';

import Meta from 'client/common/components/meta';
import PopupsContainer from 'client/common/components/popups-container';
import { IMetrikaOptions, YMInitializer } from 'client/common/components/yandex-metrika';
import {
    ConfigContextProvider,
    SettingsContextProvider,
    UserContextProvider
} from 'client/common/contexts';
import { useComponents } from 'client/common/hooks';
import { IAppData } from 'client/common/types';

import configureStore, { IAppState } from 'store';
import cn from 'utils/cn';

import { usePageRenderers } from './page-renderers';

import './index.css';

import 'client/common/styles/fonts.css';
import 'client/common/styles/meta.css';

import 'client/common/styles/variables.css';

export interface IAppProps {
    data: IAppData;
    state: IAppState;
}

const b = cn('app');

function App({ data, state }: IAppProps) {
    const store = useRef(configureStore(state));

    const {
        Footer,
        Prefooter,
        MainMenu,
        SidebarPusher,
        SidebarDimmer,
        Sidebar
    } = useComponents();

    const pageRenderers = usePageRenderers(data);

    const {
        metrikaCounters,
        metrikaCountersParams,
        settings,
        seo,
        og,
        siteSearchUrl,
        language,
        uatraits,
        faviconUrl,
        tld,
        regions,
        mainMenu,
        prefooter
    } = data;
    const { phone, text } = settings.footer;
    const phoneData = { phone, text };

    return (
        <div className={b({}, [cnTheme(theme)])}>
            <Provider store={store.current}>
                <SettingsContextProvider data={data}>
                    <ConfigContextProvider data={data}>
                        <UserContextProvider data={data}>
                            <Helmet>
                                <html lang={language} />
                                <link rel="shortcut icon" href={faviconUrl} />
                            </Helmet>
                            <Meta uatraits={uatraits} seo={seo} og={og} />

                            <MainMenu menu={mainMenu} siteSearchUrl={siteSearchUrl} />

                            <Sidebar
                                menu={mainMenu}
                                tld={tld}
                                regions={regions}
                                phoneData={phoneData}
                                siteSearchUrl={siteSearchUrl}
                                />
                            <SidebarDimmer />
                            <SidebarPusher>
                                <Switch>
                                    <Route
                                        exact
                                        path="/solutions"
                                        render={pageRenderers.solutions}
                                        />
                                    <Route
                                        exact
                                        path="/solutions/:section"
                                        render={pageRenderers.solutions}
                                        />
                                    <Route
                                        path="/solutions/cases/:slug"
                                        render={pageRenderers.solution}
                                        />
                                    <Route
                                        path="/solutions/practicums/:slug"
                                        render={pageRenderers.solution}
                                        />
                                    <Route
                                        path="/solutions/analytics/:slug"
                                        render={pageRenderers.solution}
                                        />
                                    <Route
                                        path="/solutions/stories/:slug"
                                        render={pageRenderers.solution}
                                        />
                                    <Route
                                        exact
                                        path="/analytics-now"
                                        render={pageRenderers.solutions}
                                        />
                                    <Route
                                        exact
                                        path="/analytics-now/:slug"
                                        render={pageRenderers.solution}
                                        />
                                    <Route
                                        path="/contact/agencies/:slug/"
                                        render={pageRenderers.agency}
                                        />
                                    <Route
                                        path="/contact/agencies/"
                                        render={pageRenderers.agencies}
                                        />
                                </Switch>
                                <Prefooter sections={prefooter} />
                                <Footer />
                            </SidebarPusher>

                        </UserContextProvider>
                    </ConfigContextProvider>
                </SettingsContextProvider>
                <PopupsContainer />
            </Provider>
            <YMInitializer
                accounts={metrikaCounters.map(counter => Number(counter.id))}
                options={prepareMetikaOptions()}
                />
            <script src={data.shareScript} />
            {data.mapsApi && <script id="ymaps-loader" src={data.mapsApi} />}
        </div>
    );

    function prepareMetikaOptions() {
        return metrikaCounters.reduce((acc: IMetrikaOptions, counter) => {
            acc[counter.id] = { ...metrikaCountersParams, ...counter.params };

            return acc;
        }, {});
    }
}

export default App;
