import React, { ReactNode } from 'react';

import { useConfig } from 'client/common/hooks';

import cn from 'utils/cn';
import { humanizeDate } from 'utils/date';

import './index.css';

export interface ICardProps {
    classNames?: string[];
    view?: 'default' | 'agency';
    title: string;
    url: string;
    date: Date | string | number | null;
    mark?: string;
    tags?: ReactNode;
    theme?: string;
    companySize?: string;
    backgroundImage?: string;
    backgroundImageView?: string;
    isImportant?: boolean;
    cardType?: string;
    isDatesHidden?: boolean;
}

export const b = cn('card');

function Card({
    classNames = [],
    view = 'default',
    title,
    url,
    date,
    mark,
    tags = null,
    theme,
    companySize,
    backgroundImage,
    backgroundImageView,
    isImportant,
    cardType,
    isDatesHidden
}: ICardProps) {
    const { language } = useConfig();

    const mods = {
        image: backgroundImage ? 'yes' : undefined,
        imageView: backgroundImageView,
        important: isImportant,
        type: cardType,
        view
    };

    const backgroundStyle = backgroundImage
        ? { backgroundImage: `url(${backgroundImage})` }
        : {};

    return (
        <div className={b(mods, classNames)}>
            <div className={b('background')} style={backgroundStyle} />
            {mark && (
                <div className={b('mark')}>
                    {mark}
                </div>
            )}
            {tags && (
                <div className={b('tags')}>
                    {tags}
                </div>
            )}
            {view !== 'agency' && (
                <div className={b('theme-wrapper')}>
                    {theme && (
                        <div
                            className={b('theme', { companySize })}
                            dangerouslySetInnerHTML={{ __html: theme }}
                            />
                    )}
                </div>
            )}
            <h2 className={b('title')}>
                <a
                    href={url}
                    dangerouslySetInnerHTML={{ __html: title }}
                    />
            </h2>
            {date && !isDatesHidden && (
                <div className={b('date')}>
                    <time dateTime={new Date(date).toISOString()} itemProp="dateModified">
                        {humanizeDate(date, 'full', language)}
                    </time>
                </div>
            )}
        </div>
    );
}

interface ITagProps {
    name: string;
    url?: string;
    onClick?: () => void;
}

function Tag({ name, url, onClick }: ITagProps) {
    if (url) {
        return (
            <a
                className={b('tag')}
                href={url}
                >
                {name}
            </a>
        );
    }

    return (
        <button
            className={b('tag')}
            onClick={onClick}
            >
            {name}
        </button>
    );
}

Card.Tag = Tag;

export default Card;
