import React, { useState } from 'react';

import { useComponents } from 'client/common/hooks';

import cn from 'utils/cn';
import { bindKeyset } from 'utils/i18n';

import './index.css';

export interface IExpandableTextProps {
    className?: string;
    text: string;
    shortenedTextMaxLength: number;
}

export const b = cn('expandable-text');
const i18n = bindKeyset('common');

const TOKEN_TYPES = [
    {
        type: 'html-entity',
        regExpText: '&[a-zA-Z]+;|&#\\d+;',
        regExp: new RegExp('&[a-zA-Z]+;|&#\\d+;'),
        getLength: () => 1
    },
    {
        type: 'link',
        regExpText: '<a\\s[^]*?href=(?:"|\').*?(?:"|\')[^]*?>[^]*?<\\/a>',
        regExp: new RegExp('<a\\s[^]*?href=((?:"|\').*?(?:"|\'))[^]*?>([^]*?)<\\/a>'),
        getLength: (_: string, match: RegExpMatchArray) => {
            const [,, text] = match;

            return text.length;
        }
    }
];

function ExpandableText({ className, text, shortenedTextMaxLength }: IExpandableTextProps) {
    const preparedText = prepareText(text);
    const shortenedText = getShortenedText(preparedText, shortenedTextMaxLength);
    const isShortened = shortenedText !== preparedText;

    const [isExpanded, setIsExpanded] = useState(!isShortened);
    const { HtmlContent } = useComponents();

    function onExpandButtonClick() {
        setIsExpanded(true);
    }

    return (
        <div className={b({}, [className])}>
            {isExpanded ?
                (
                    <div className={b('full')}>
                        <HtmlContent
                            key="full-text"
                            className={b('text')}
                            theme="light"
                            content={preparedText}
                            />
                    </div>
                ) :
                (
                    <div className={b('shortened')}>
                        <HtmlContent
                            key="short-text"
                            className={b('text')}
                            theme="light"
                            content={shortenedText}
                            />
                        <div
                            className={b('expand-button')}
                            onClick={onExpandButtonClick}
                            >
                            {i18n('show-full')}
                        </div>
                    </div>
                )
            }
        </div>
    );
}

function prepareText(text: string) {
    return text.replace(
        /(\s|^)(https?:\/\/\S+)/g,
        '$1<a href="$2" rel="noopener" target="_blank">$2</a>'
    );
}

function getShortenedText(text: string, shortenedTextMaxLength: number) {
    const tokens = parseTokens(text);
    let shortenedText = '';
    let shortenedTextLength = 0;

    for (const token of tokens) {
        const newLength = shortenedTextLength + token.length;

        if (newLength <= shortenedTextMaxLength) {
            shortenedText = shortenedText + token.text;
            shortenedTextLength = newLength;

            continue;
        }

        if (token.type === 'text') {
            const maxAdditionalTextLength = shortenedTextMaxLength - shortenedTextLength;

            shortenedText += token.text.slice(0, maxAdditionalTextLength);
        }

        shortenedText = `${shortenedText.replace(/\s+$/, '')}…`;

        break;
    }

    return shortenedText.replace(/(\r\n|\n|\r)/g, ' ');
}

function parseTokens(text: string) {
    const tokensRegExpText = TOKEN_TYPES
        .map(tokenType => {
            return tokenType.regExpText;
        })
        .join('|');

    const tokensRegExp = new RegExp(`(${tokensRegExpText})`, 'g');

    const tokens = text
        .split(tokensRegExp)
        .map(token => {
            const tokenType = TOKEN_TYPES.find(item => token.match(item.regExp));

            if (tokenType) {
                const match = token.match(tokenType.regExp);

                if (match) {
                    return {
                        type: tokenType.type,
                        text: token,
                        length: tokenType.getLength(token, match)
                    };
                }
            }

            return {
                type: 'text',
                text: token,
                length: token.length
            };
        });

    return tokens;
}

export default ExpandableText;
