import { IClassNameProps } from '@bem-react/core';
import { ItemGroup } from '@yandex-lego/components/Menu';
import React, { ChangeEvent, useCallback } from 'react';

import { useLegoComponents } from 'client/common/hooks';
import { IFilter } from 'client/common/types';
import cn from 'utils/cn';

import './index.css';

export interface IFiltersSelectProps extends IClassNameProps {
    disabled?: string[];
    items: IFilter[];
    value?: string[];
    placeholder: string;

    add(filter: string): void;
}

const b = cn('filters-select');

export default function FiltersSelect(props: IFiltersSelectProps) {
    const { Select } = useLegoComponents();
    const { add, className, items, placeholder, value = [], disabled = [] } = props;

    const handleChange = useCallback((event: ChangeEvent<HTMLSelectElement>) => {
        add(event.target.value);
    }, [add]);

    const options = convertToOptions(items, value, disabled);
    const isDisabled = !items.length || items.every(
        item => value.includes(item.slug) || disabled.includes(item.slug)
    );

    if (isDisabled && value.length === 0) {
        return null;
    }

    return (
        <div className={b({}, [className])}>
            <Select
                disabled={isDisabled}
                onChange={handleChange}
                options={options}
                placeholder={placeholder}
                size="m"
                theme="normal"
                />
        </div>
    );
}

function convertToOptions(items: IFilter[], excluded: string[], disabled: string[]) {
    const groupToIndex: Record<string, number> = {};

    return items
        .filter(item => !excluded.includes(item.slug))
        .reduce((groups, item) => {
            const title = item.group || '';
            const index = groupToIndex[title];

            const option = {
                content: item.name,
                value: item.slug,
                disabled: disabled.includes(item.slug)
            };

            if (index === undefined) {
                groupToIndex[title] = groups.length;

                return groups.concat({
                    items: [option],
                    title
                });
            }

            groups[index].items = groups[index].items.concat(option);

            return groups;
        }, [] as ItemGroup[]);
}
