import url from 'url';

import React, { useEffect, useRef } from 'react';

import cn from 'utils/cn';
import { reachGoal } from 'utils/metrika';

const b = cn('form-iframe');

interface IFormIframeProps {
    src: string;
    className?: string;
    initialWidth?: number | string;
    initialHeight?: number | string;
    metrikaGoalId?: string;
    adaptive?: boolean;
    query?: Record<string, string | number>;
}

interface IPostMessageData {
    message: string;
    name?: string;
    'iframe-height'?: number;
}

function getFormName(src: string) {
    const match = src.match(/\d+/);

    const name = match ? match[0] : '';

    return name;
}

function constructFormSrc(src: string, query: IFormIframeProps['query'] = {}) {
    query.iframe = 1;

    return url.format({
        ...url.parse(src),
        query
    });
}

function FormIframe({
    src,
    className,
    metrikaGoalId,
    adaptive,
    initialWidth = '100%',
    initialHeight = '550px',
    query
}: IFormIframeProps) {
    const iframeRef = useRef<HTMLIFrameElement>(null);

    const name = getFormName(src);
    const formSrc = constructFormSrc(src, query);

    useEffect(() => {
        function onPingMessage(data: IPostMessageData, source: WindowProxy) {
            // Iframe anti ClickJacking
            if (source && data.message === 'ping') {
                source.postMessage('pong', '*');
            }
        }

        function isCurrentFormData(data: IPostMessageData) {
            return data.name && data.name === name;
        }

        function onResizeMessage(data: IPostMessageData) {
            // Через postMessage с интервалом в 300мс отправляется высота страницы
            // Меняем высоту iframe
            if (data['iframe-height'] && adaptive) {
                const height = data['iframe-height'];

                const iframe = iframeRef.current;

                if (iframe) {
                    iframe.style.height = `${height}px`;
                }
            }
        }

        function onSendMessage(data: IPostMessageData) {
            if (data.message === 'sent' && metrikaGoalId) {
                reachGoal(metrikaGoalId);
            }
        }

        function onIframeMessage({ data, source }: MessageEvent) {
            try {
                data = JSON.parse(data);
            } catch (err) {
                return;
            }

            onPingMessage(data, source as WindowProxy);

            if (!isCurrentFormData(data)) {
                return;
            }

            onResizeMessage(data);

            onSendMessage(data);
        }

        window.addEventListener('message', onIframeMessage);

        return () => window.removeEventListener('message', onIframeMessage);
    }, [name, metrikaGoalId, adaptive]);

    return (
        <iframe
            ref={iframeRef}
            src={formSrc}
            name={name}
            className={b({}, [className])}
            allowFullScreen={false}
            frameBorder={0}
            style={{
                width: initialWidth,
                height: initialHeight
            }}
            />
    );
}

export default FormIframe;
