import { withBemMod } from '@bem-react/core';
import React, { createRef, useCallback, useEffect } from 'react';
import ReactSwipe from 'react-swipe';

import { useComponents, useConfig } from 'client/common/hooks';
import { ILevelMedia } from 'client/common/types';
import { isElementVisible } from 'utils/helper';
import i18n from 'utils/i18n';

import { b, ILevelProps } from '..';

import './level_type_carousel.css';

export interface ILevelTypeCarouselProps extends Omit<ILevelMedia, 'type'> {
    type?: 'carousel';
}

export const withLevelTypeCarousel = withBemMod<ILevelTypeCarouselProps, ILevelProps>(
    b(),
    { type: 'carousel' },
    (BaseLevel: React.FC<ILevelProps>) => props => {
        const { blocks, ...otherProps } = props;

        const swipeRef = createRef<ReactSwipe>();
        const contentRef = createRef<HTMLDivElement>();

        const { isMobile } = useConfig().uatraits;
        const { HtmlContent } = useComponents();

        const prev = useCallback(() => swipeRef.current && swipeRef.current.prev(), [swipeRef]);
        const next = useCallback(() => swipeRef.current && swipeRef.current.next(), [swipeRef]);

        useEffect(() => {

            function handleKeyUp(event: KeyboardEvent) {
                if (!contentRef.current || !isElementVisible(contentRef.current)) {
                    return;
                }

                if (event.code === 'ArrowLeft') {
                    prev();
                } else if (event.code === 'ArrowRight') {
                    next();
                }
            }

            if (!isMobile) {
                window.addEventListener('keyup', handleKeyUp);
            }

            return () => {
                if (!isMobile) {
                    window.removeEventListener('keyup', handleKeyUp);
                }
            };
        }, [contentRef, isMobile, next, prev]);

        return (
            <BaseLevel {...otherProps}>
                <div ref={contentRef} className={b('content')}>
                    <ReactSwipe
                        className={b('swipe')}
                        ref={swipeRef}
                        >
                        {blocks && blocks.map(({ image, bottomText }, id) =>
                            (
                                <div
                                    key={image}
                                    className={b('block')}
                                    >
                                    {image && (
                                        <div className={b('image-wrapper')}>
                                            <img
                                                className={b('image')}
                                                src={image}
                                                />
                                        </div>
                                    )}
                                    <div className={b('row')}>
                                        {bottomText && (
                                            <HtmlContent
                                                className={b('bottom-text')}
                                                content={bottomText}
                                                />
                                        )}
                                        <div
                                            className={b('slide-info')}
                                            >
                                            {`${id + 1} ${i18n({
                                                key: 'swipe.slideInfo',
                                                keyset: 'common'
                                            })} ${blocks.length}`}
                                        </div>
                                    </div>
                                </div>
                            ))}
                    </ReactSwipe>
                    <div className={b('buttons')}>
                        <div className={b('button-prev')} onClick={prev} />
                        <div className={b('button-next')} onClick={next} />
                    </div>
                </div>
            </BaseLevel>
        );
    }
);
