import { classnames } from '@bem-react/classnames';
import { withBemMod } from '@bem-react/core';
import React, { useEffect, useRef } from 'react';

import FormIframe from 'client/common/components/form-iframe';
import { useComponents, useLegoComponents } from 'client/common/hooks';
import { ILevelCta } from 'client/common/types';
import { detectScrollEnd, reachGoal } from 'utils/metrika';

import { s } from 'utils/findCustomStyle';

import { b, ILevelProps } from '..';

import './level_type_cta.css';

export interface ILevelTypeCtaProps extends Omit<ILevelCta, 'type'> {
    type?: 'cta';
}

export const withLevelTypeCta = withBemMod<ILevelTypeCtaProps, ILevelProps>(
    b(),
    { type: 'cta' },
    (BaseLevel: React.FC<ILevelProps>) => props => {
        const levelRef = useRef<HTMLDivElement>(null);

        const { blocks, scrollEndGoalId, className, customStyles, ...otherProps } = props;

        if (scrollEndGoalId) {
            useEffect(() => {
                function scrollHandler() {
                    detectScrollEnd(levelRef.current!, scrollEndGoalId!, scrollHandler);
                }

                window.addEventListener('scroll', scrollHandler);

                return () => window.removeEventListener('scroll', scrollHandler);
            });
        }

        const { Button } = useLegoComponents();
        const { HtmlContent } = useComponents();

        const levelClass =
            blocks.length === 2
                ? classnames(className, b({ size: 2 }))
                : className;

        return (
            <BaseLevel className={levelClass} innerRef={levelRef} {...otherProps}>
                <div className={b('content')}>
                    {blocks.map(({
                        title,
                        description,
                        buttonText,
                        buttonUrl,
                        metrikaGoalId,
                        formUrl
                    }) => {
                        const isFormBlock = Boolean(formUrl);

                        return (
                            <div key={title} className={b('block')} style={s('Level', customStyles, ['background'])}>
                                <h3
                                    className={b('block-title')}
                                    dangerouslySetInnerHTML={{ __html: title }}
                                    style={s('Level', customStyles, ['color'])}
                                    />
                                {description && (
                                <HtmlContent
                                    className={b('block-description')}
                                    content={description}
                                    style={s('Level', customStyles, ['color'])}
                                    />
                                )}
                                {!isFormBlock && buttonText && buttonUrl && (
                                    <Button
                                        className={b('button')}
                                        theme="action"
                                        size="m"
                                        type="link"
                                        url={buttonUrl}
                                        target="_blank"
                                        rel="noopener noreferrer"
                                        onClick={() => metrikaGoalId && reachGoal(metrikaGoalId)}
                                        >
                                        {buttonText}
                                    </Button>
                                )}
                                {isFormBlock && (
                                    <FormIframe
                                        src={formUrl!}
                                        className={b('form')}
                                        adaptive
                                        />
                                )}
                            </div>
                        );
                    })}
                </div>
            </BaseLevel>
        );
    }
);
