import { withBemMod } from '@bem-react/core';
import React, { useCallback } from 'react';

import FiltersSelect from 'client/common/components/filters-select';
import { useComponents, useFilters, useLegoComponents } from 'client/common/hooks';
import { IFilter, TFiltersByCategory, TFiltersCategory } from 'client/common/types';
import i18n from 'utils/i18n';

import { b, ILevelProps } from '..';

import './level_type_filters.css';

export interface ILevelTypeFiltersProps extends Omit<ILevelTypeFiltersProps, 'type'> {
    type?: 'filters';

    hasMore?: TFiltersByCategory;
    groups: Record<TFiltersCategory, IFilter[]>;
    order: TFiltersCategory[];

    text: string;
    title: string;
}

export const withLevelTypeFilters = withBemMod<ILevelTypeFiltersProps, ILevelProps>(
    b(),
    { type: 'filters' },
    (BaseLevel: React.FC<ILevelProps>) => props => {
        const {
            hasMore = {} as TFiltersByCategory,
            groups,
            order,
            text,
            title,
            ...otherProps
        } = props;

        const { Button, Icon } = useLegoComponents();
        const { HtmlContent } = useComponents();
        const { add, value: filters, remove, removeAll } = useFilters();

        const disabled = convertToDisabled(groups, hasMore);

        const CloseIcon = useCallback((className: string) => (
            <Icon
                className={b('close-icon', [className])}
                type="close"
                />
        ), []);

        const addHandler = (category: TFiltersCategory) => (item: string) => {
            add(category, item);
        };

        const removeHandler = (category: TFiltersCategory, item: string) => () => {
            remove(category, item);
        };

        let hasAnyActiveFilter = false;

        return (
            <BaseLevel {...otherProps}>
                <h1 className={b('title')}>{title}</h1>
                <HtmlContent className={b('text')} content={text} />

                {order.map((category: TFiltersCategory) => (
                    <FiltersSelect
                        key={category}
                        className={b('select')}
                        add={addHandler(category)}
                        disabled={disabled[category]}
                        items={groups[category]}
                        placeholder={i18n({ keyset: 'solutions', key: category })}
                        value={filters[category] || []}
                        />
                ))}

                <div className={b('active-filters')}>
                    {order.map((category: TFiltersCategory) =>
                        (filters[category] || []).map((filterSlug: string) => {
                            const group = groups[category];
                            const filter = group.find(item => item.slug === filterSlug);

                            if (!filter) {
                                return null;
                            }

                            hasAnyActiveFilter = true;

                            return (
                                <Button
                                    className={b('active-filter')}
                                    key={filter.slug}
                                    iconRight={CloseIcon}
                                    onClick={removeHandler(category, filter.slug)}
                                    size="m"
                                    theme="normal"
                                    >
                                    {filter.name}
                                </Button>
                            );
                        })
                    )}
                    {hasAnyActiveFilter && (
                        <Button
                            className={b('clear-all')}
                            onClick={removeAll}
                            size="m"
                            theme="normal"
                            >
                            {i18n({ keyset: 'filters', key: 'clear' })}
                        </Button>
                    )}
                </div>
            </BaseLevel>
        );
    }
);

function convertToDisabled(
    groups: Record<TFiltersCategory, IFilter[]>,
    hasMore: TFiltersByCategory
) {
    return Object.keys(groups)
        .reduce((acc, category: TFiltersCategory) => ({
            ...acc,
            [category]: groups[category].reduce((disabledInCategory: string[], { slug }: IFilter) =>
                ((hasMore && hasMore[category]) || []).includes(slug)
                    ? disabledInCategory
                    : disabledInCategory.concat(slug)
            , [])
        }), {}) as TFiltersByCategory;
}
