import { withBemMod } from '@bem-react/core';
import React, { Fragment, useContext } from 'react';

import { IExtendedImage } from 'client/common/components/extended-image/ExtendedImage.type';
import OverlayScroll from 'client/common/components/overlay-scroll';
import { ConfigContext } from 'client/common/contexts';
import { useComponents } from 'client/common/hooks';
import { ILevelMedia } from 'client/common/types';

import { b, ILevelProps } from '..';

import './level_type_media.css';

export interface ILevelMediaProps extends Omit<ILevelMedia, 'type'> {
    type?: 'media';
}

function constructNElements(n: number, Element: React.FC<{}>) {
    const elements = [];

    for (let i = 0; i < n; i++) {
        elements.push(<Element key={i} />);
    }

    return elements;
}

export const withLevelTypeMedia = withBemMod<ILevelMediaProps, ILevelProps>(
    b(),
    { type: 'media' },
    (BaseLevel: React.FC<ILevelProps>) => props => {

        const { imageStyle, caption: generalCaption, blocks, ...otherProps } = props;
        const blocksLength = blocks.length;

        const header: string[] = [];
        const body: Array<{image: string|undefined, extendedImage: IExtendedImage|undefined, caption: string|undefined }> = [];
        const captions: string[] = [];
        const footer: string[] = [];

        blocks.forEach(({ topText, image, extendedImage, caption, bottomText }) => {
            header.push(topText || '');
            body.push({ image, extendedImage, caption });
            captions.push(caption || '');
            footer.push(bottomText || '');
        });

        const scrollableMod = { scrollable: blocksLength > 1 };
        const sizeMod = { size: blocksLength };

        const { uatraits } = useContext(ConfigContext);
        const { width = 0, height = 0 } = imageStyle || {};
        const style = !uatraits.isMobile && blocksLength === 1 ? {
            width: width || (height ? 'auto' : '100%'),
            height: height || (width ? 'auto' : undefined)
        } : {};

        const { HtmlContent, ExtendedImage } = useComponents();

        /* eslint-disable react/no-array-index-key */
        return (
            <BaseLevel {...otherProps}>
                <Fragment>
                    {blocksLength >= 1 && (
                    <OverlayScroll className={b('content', scrollableMod)}>
                        <div className={b('blocks', sizeMod)}>
                            <div className={b('columns', sizeMod)}>
                                {constructNElements(
                                    blocksLength, () => <div className={b('column')} />
                                )}
                            </div>
                            <div className={b('blocks-header')}>
                                <div className={b('row')}>
                                    {header.map((topText, id) => (
                                        <HtmlContent
                                            key={id}
                                            className={b('top-text')}
                                            content={topText}
                                            />
                                    ))}
                                </div>
                            </div>
                            <div className={b('blocks-body')}>
                                <div className={b('row')}>
                                    {body.map((imageObj, id) => (
                                        <div key={id} className={b('image')}>
                                            { imageObj?.image && <img src={imageObj.image} style={style} /> }
                                            {
                                                !imageObj?.image &&
                                                imageObj?.extendedImage &&
                                                imageObj?.extendedImage &&
                                                <ExtendedImage extendedImage={imageObj.extendedImage} caption={imageObj?.caption} style={style} className={b('image')} />

                                            }
                                        </div>
                                    ))}
                                </div>
                            </div>
                            <div className={b('blocks-footer')}>
                                <div className={b('row')}>
                                    {captions.map((caption, id) => (
                                        <HtmlContent
                                            key={id}
                                            className={b('caption')}
                                            content={caption}
                                            />
                                    ))}
                                </div>
                                <div className={b('row')}>
                                    {footer.map((bottomText, id) => (
                                        <HtmlContent
                                            key={id}
                                            className={b('bottom-text')}
                                            content={bottomText}
                                            />
                                    ))}
                                </div>
                            </div>
                        </div>
                    </OverlayScroll>)}
                    {generalCaption && <HtmlContent className={b('caption')} content={generalCaption} />}
                </Fragment>
            </BaseLevel>
        );
    }
);
