import { ComponentRegistryConsumer } from '@bem-react/di';
import React, { Component, createRef } from 'react';

import { IMenuItem, ISideMenu } from 'client/common/types';
import Search from 'client/platforms/desktop/components/search';
import cn from 'utils/cn';
import debounce from 'utils/debounce';
import i18n from 'utils/i18n';

import './index.css';

export interface IMainMenuProps {
    hasSearch: boolean;
    siteSearchUrl?: string;
    isOpen: boolean;
    menuIndexChain: number[];
    menu: ISideMenu;
    scrollableHead?: boolean;
    toggleSidebar(): void;
}

const cnMainMenu = cn('main-menu');

class MainMenu extends Component<IMainMenuProps> {
    private readonly headRef = createRef<HTMLElement>();

    private autoOpenWidth = 1240;
    private prevViewPortWidth: number;

    private prevWinPos: number;
    private maxHeaderHeight: number;

    normalizeMargin = (margin: number) => {
        if (margin > 0) {
            return 0;
        }

        if (margin < this.maxHeaderHeight) {
            return this.maxHeaderHeight;
        }

        return margin;
    }

    componentDidMount() {
        const { scrollableHead, isOpen } = this.props;

        if (this.headRef.current) {
            this.maxHeaderHeight = -this.headRef.current.clientHeight;
            this.prevWinPos = window.pageYOffset;
        }

        if (scrollableHead) {
            window.addEventListener('scroll', this.onScroll);

            this.prevViewPortWidth = window.innerWidth;
            window.addEventListener('resize', debounce(this.onResize, 300, true));

            this.tryOpenMenu(isOpen);
        }
    }

    componentWillUnmount() {
        window.removeEventListener('scroll', this.onScroll);
    }

    render() {
        const { hasSearch, siteSearchUrl, isOpen, menu, menuIndexChain } = this.props;

        const mods = isOpen ? { sidebar: 'opened' } : {};

        const switchText = i18n({
            key: 'menu',
            keyset: 'menu'
        });

        const breadcrumbs: Array<IMenuItem | ISideMenu> = [menu];

        for (let i = 0; i < menuIndexChain.length; i++) {
            const lastIndex = breadcrumbs.length - 1;
            const { items } = breadcrumbs[lastIndex];

            if (items && items.length > 0) {
                breadcrumbs.push(items[menuIndexChain[i]]);
            }
        }

        return (
            <ComponentRegistryConsumer id={'app'}>
                {({ Header }) => (
                    <div className={cnMainMenu(mods)}>
                        <Header headerRef={this.headRef} />
                        <div className={cnMainMenu('head')}>
                            <div className={cnMainMenu('switch')} onClick={this.onClick}>
                                <div className={cnMainMenu('switch-icon')}>
                                    <div className={cnMainMenu('switch-close')} />
                                    <div className={cnMainMenu('switch-open')} />
                                </div>
                                <span className={cnMainMenu('switch-text')}>{switchText}</span>
                            </div>
                            <div
                                className={cnMainMenu('breadcrumbs')}
                                itemType="http://schema.org/BreadcrumbList"
                                itemScope
                                >
                                {breadcrumbs.map(({ title, url }, index) => (
                                    <div
                                        key={title}
                                        className={cnMainMenu('breadcrumbs-item')}
                                        itemProp="itemListElement"
                                        itemType="http://schema.org/ListItem"
                                        itemScope
                                        >
                                        <a href={url} dangerouslySetInnerHTML={{ __html: title }} />
                                        <meta itemProp="position" content={index.toString(10)} />
                                        <meta itemProp="name" content={title} />
                                        <meta itemProp="item" content={url} />
                                    </div>
                                ))}
                            </div>
                            {/* TODO: не забыть про cta */}
                            {/* https://l7test.yandex.ru/adv/products/classified/market */}
                            {hasSearch && (
                                <div className={cnMainMenu('right-content')}>
                                    <div className={cnMainMenu('search')}>
                                        <Search siteSearchUrl={siteSearchUrl!} type="menu" />
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>
                )}
            </ComponentRegistryConsumer>
        );
    }

    private onScroll = () => {
        const pageYDifference = window.pageYOffset - this.prevWinPos;
        const headMargin = parseInt(this.headRef.current!.style.marginTop! || '0', 10);
        const slideHeadUp = pageYDifference < 0 && headMargin < 0;
        const headMarginOverflow = headMargin > this.maxHeaderHeight;
        const slideHeadDown = pageYDifference > 0 && headMarginOverflow && window.pageYOffset > 0;
        const preparingMargin = headMargin - pageYDifference;
        const newHeadMargin = this.normalizeMargin(preparingMargin);

        if (slideHeadUp || slideHeadDown) {
            this.headRef.current!.style.marginTop = `${newHeadMargin}px`;
        }

        this.prevWinPos = window.pageYOffset;
    }

    private onClick = () => this.props.toggleSidebar();

    private tryOpenMenu = (isMenuOpened: boolean) => {
        if (isMenuOpened) {
            return;
        }

        if (this.prevViewPortWidth >= this.autoOpenWidth) {
            this.props.toggleSidebar();
        } else if (!this.props.scrollableHead && !isMenuOpened) {
            this.props.toggleSidebar();
        }
    }

    /**
     * При ресайзе окна открываем меню только в случае увеличения
     * @private
     */
    private onResize = () => {
        const currentWidth = window.innerWidth;

        if (
            !this.props.isOpen
            && this.prevViewPortWidth < this.autoOpenWidth
            && currentWidth >= this.autoOpenWidth
        ) {
            this.prevViewPortWidth = currentWidth;
            this.tryOpenMenu(false);
        }
    }
}

export default MainMenu;
